/*
 * waypoints.c -- GPS protocol waypoint handling.
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/waypoints.c,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 21:33 Apr 24 2000 kivinen
 *	  Last Modification : 12:38 Aug 19 2003 kivinen
 *	  Last check in     : $Date: 2003/08/18 20:09:33 $
 *	  Revision number   : $Revision: 1.9 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.188
 *	  Edit time	    : 97 min
 *
 *	  Description       : GPS protocol waypoint handling
 *
 *	  $Log: waypoints.c,v $
 *	  Revision 1.9  2003/08/18 20:09:33  kivinen
 *	  	Added support for GPS V. Fixed packet size and actual size
 *	  	comparisions.
 *
 *	  Revision 1.8  2001/08/23 15:38:38  kivinen
 *	  	Fixed bug in d108 format, the proximity field was missing.
 *
 *	  Revision 1.7  2000/08/05 00:44:00  kivinen
 *	  	Updated to new interface.
 *
 *	  Revision 1.6  2000/07/26 17:08:36  kivinen
 *	  	Fixed typo.
 *
 *	  Revision 1.5  2000/07/21 22:16:30  kivinen
 *	  	Moved stuff to data.c.
 *
 *	  Revision 1.4  2000/07/12 22:19:57  kivinen
 *	  	Final upload support.
 *
 *	  Revision 1.3  2000/07/06 23:07:50  kivinen
 *	  	Added initial upload support.
 *
 *	  Revision 1.2  2000/04/30 01:03:31  kivinen
 *	  	Updated to rev 03 document.
 *
 *	  Revision 1.1  2000/04/29 16:41:01  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#include "tgps.h"
#include "packet.h"
#include "data.h"

int tgps_wpt_data_in_d100(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d400(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d101(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d102(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d103(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d403(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d104(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d105(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d106(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d107(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d108(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d109(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d150(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d450(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d151(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d152(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d154(Tgps conn, TgpsPacket packet, TgpsWaypoint w);
int tgps_wpt_data_in_d155(Tgps conn, TgpsPacket packet, TgpsWaypoint w);

TgpsPacket tgps_wpt_data_out_d100(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d101(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d102(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d103(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d104(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d105(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d106(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d107(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d108(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d109(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d150(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d151(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d152(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d154(Tgps conn, TgpsWaypoint w);
TgpsPacket tgps_wpt_data_out_d155(Tgps conn, TgpsWaypoint w);

#define TGPS_UPLOAD_DATA(conn) \
  ((conn->upload & 0xff) == TGPS_CMND_TRANSFER_RTE(conn) ? \
    TGPS_PID_RTE_WPT_DATA(conn) : \
    ((conn->upload & 0xff) == TGPS_CMND_TRANSFER_PRX(conn) ? \
      TGPS_PID_PRX_WPT_DATA(conn) : \
      TGPS_PID_WPT_DATA(conn)))

/* Process version 100 waypoint packet */
int tgps_wpt_data_in_d100(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_LONGWORD, &w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 400 proximity waypoint packet */
int tgps_wpt_data_in_d400(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_LONGWORD, &w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, &w->proximity,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 101 waypoint packet */
int tgps_wpt_data_in_d101(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_LONGWORD, &w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, &w->proximity,
		     TGPS_FORMAT_BYTE, &w->smbl,
		     TGPS_FORMAT_END);
  w->smbl_type = TGPS_GARMIN_SYMBOLS_FULL;
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 102 waypoint packet */
int tgps_wpt_data_in_d102(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_LONGWORD, &w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, &w->proximity,
		     TGPS_FORMAT_INT, &w->smbl16,
		     TGPS_FORMAT_END);
  w->smbl_type = TGPS_GARMIN_SYMBOLS_FULL;
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 103 waypoint packet */
int tgps_wpt_data_in_d103(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_LONGWORD, &w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_BYTE, &w->smbl,
		     TGPS_FORMAT_BYTE, &w->display,
		     TGPS_FORMAT_END);
  w->smbl_type = TGPS_GARMIN_SYMBOLS_16;
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 403 waypoint packet */
int tgps_wpt_data_in_d403(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_LONGWORD, &w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_BYTE, &w->smbl,
		     TGPS_FORMAT_BYTE, &w->display,
		     TGPS_FORMAT_FLOAT, &w->proximity,
		     TGPS_FORMAT_END);
  w->smbl_type = TGPS_GARMIN_SYMBOLS_16;
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 104 waypoint packet */
int tgps_wpt_data_in_d104(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_LONGWORD, &w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, &w->proximity, 
		     TGPS_FORMAT_INT, &w->smbl16,
		     TGPS_FORMAT_BYTE, &w->display,
		     TGPS_FORMAT_END);
  w->smbl_type = TGPS_GARMIN_SYMBOLS_FULL;
  if (w->display == 0)
    w->display = 1;
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 105 waypoint packet */
int tgps_wpt_data_in_d105(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_INT, &w->smbl16,
		     TGPS_FORMAT_STRING, &w->wpt_ident,
		     TGPS_FORMAT_END);
  w->smbl_type = TGPS_GARMIN_SYMBOLS_FULL;
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 106 waypoint packet */
int tgps_wpt_data_in_d106(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_BYTE, &w->class,
		     TGPS_FORMAT_BYTE_ARRAY, &w->subclass, 13,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_INT, &w->smbl16,
		     TGPS_FORMAT_STRING, &w->wpt_ident,
		     TGPS_FORMAT_STRING, &w->lnk_ident,
		     TGPS_FORMAT_END);
  w->smbl_type = TGPS_GARMIN_SYMBOLS_FULL;
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 107 waypoint packet */
int tgps_wpt_data_in_d107(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_LONGWORD, &w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_BYTE, &w->smbl,
		     TGPS_FORMAT_BYTE, &w->display,
		     TGPS_FORMAT_FLOAT, &w->proximity,
		     TGPS_FORMAT_BYTE, &w->color,
		     TGPS_FORMAT_END);
  w->smbl_type = TGPS_GARMIN_SYMBOLS_16;
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 108 waypoint packet */
int tgps_wpt_data_in_d108(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_BYTE, &w->class,
		     TGPS_FORMAT_BYTE, &w->color,
		     TGPS_FORMAT_BYTE, &w->display,
		     TGPS_FORMAT_BYTE, &w->attribute,
		     TGPS_FORMAT_INT, &w->smbl16,
		     TGPS_FORMAT_BYTE_ARRAY, &w->subclass, 18,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_FLOAT, &w->altf,
		     TGPS_FORMAT_FLOAT, &w->dpth,
		     TGPS_FORMAT_FLOAT, &w->proximity,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_STRING, &w->wpt_ident,
		     TGPS_FORMAT_STRING, &w->comment_var,
		     TGPS_FORMAT_STRING, &w->facility_var,
		     TGPS_FORMAT_STRING, &w->city_var,
		     TGPS_FORMAT_STRING, &w->addr_var,
		     TGPS_FORMAT_STRING, &w->cross_road_var,
		     TGPS_FORMAT_END);
  w->smbl_type = TGPS_GARMIN_SYMBOLS_FULL;
  if (w->color == -1)
    w->color = 0;
  else
    w->color += 16;

  if (w->altf >= 1.0e20)
    w->altf = TGPS_INVALID_FLOAT;
  if (w->dpth >= 1.0e20)
    w->dpth = TGPS_INVALID_FLOAT;
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 109 waypoint packet */
int tgps_wpt_data_in_d109(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_BYTE, NULL, /* Dtype */
		     TGPS_FORMAT_BYTE, &w->class,
		     TGPS_FORMAT_BYTE, &w->color,
		     TGPS_FORMAT_BYTE, &w->attribute,
		     TGPS_FORMAT_INT, &w->smbl16,
		     TGPS_FORMAT_BYTE_ARRAY, &w->subclass, 18,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_FLOAT, &w->altf,
		     TGPS_FORMAT_FLOAT, &w->dpth,
		     TGPS_FORMAT_FLOAT, &w->proximity,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_LONGWORD, &w->ete,
		     TGPS_FORMAT_STRING, &w->wpt_ident,
		     TGPS_FORMAT_STRING, &w->comment_var,
		     TGPS_FORMAT_STRING, &w->facility_var,
		     TGPS_FORMAT_STRING, &w->city_var,
		     TGPS_FORMAT_STRING, &w->addr_var,
		     TGPS_FORMAT_STRING, &w->cross_road_var,
		     TGPS_FORMAT_END);

  w->display = (w->color >> 5) & 0x3;
  w->color = w->color & 0x1f;
  w->smbl_type = TGPS_GARMIN_SYMBOLS_FULL;
  if (w->color == 0x1f)
    w->color = 0;
  else
    w->color += 16;

  if (w->altf >= 1.0e20)
    w->altf = TGPS_INVALID_FLOAT;
  if (w->dpth >= 1.0e20)
    w->dpth = TGPS_INVALID_FLOAT;
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 150 waypoint packet */
int tgps_wpt_data_in_d150(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_BYTE, &w->class,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_INT, &w->alt,
		     TGPS_FORMAT_CHAR_ARRAY, w->city, 24,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_CHAR_ARRAY, w->name, 30,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 450 waypoint packet */
int tgps_wpt_data_in_d450(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_INT, &w->idx,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_BYTE, &w->class,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_INT, &w->alt,
		     TGPS_FORMAT_CHAR_ARRAY, w->city, 24,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_CHAR_ARRAY, w->name, 30,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, &w->proximity,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 151 waypoint packet */
int tgps_wpt_data_in_d151(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;
  char unused2;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_LONGWORD, &w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, &w->proximity,
		     TGPS_FORMAT_CHAR_ARRAY, w->name, 30,
		     TGPS_FORMAT_CHAR_ARRAY, w->city, 24,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_INT, &w->alt,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_CHAR, &unused2,
		     TGPS_FORMAT_BYTE, &w->class, 
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 152 waypoint packet */
int tgps_wpt_data_in_d152(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;
  char unused2;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_LONGWORD, &w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, &w->proximity,
		     TGPS_FORMAT_CHAR_ARRAY, w->name, 30,
		     TGPS_FORMAT_CHAR_ARRAY, w->city, 24,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_INT, &w->alt,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_CHAR, &unused2,
		     TGPS_FORMAT_BYTE, &w->class, 
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 154 waypoint packet */
int tgps_wpt_data_in_d154(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;
  char unused2;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_LONGWORD, &w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, &w->proximity,
		     TGPS_FORMAT_CHAR_ARRAY, w->name, 30,
		     TGPS_FORMAT_CHAR_ARRAY, w->city, 24,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_INT, &w->alt,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_CHAR, &unused2,
		     TGPS_FORMAT_BYTE, &w->class,
		     TGPS_FORMAT_INT, &w->smbl16,
		     TGPS_FORMAT_END);
  w->smbl_type = TGPS_GARMIN_SYMBOLS_FULL;
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 155 waypoint packet */
int tgps_wpt_data_in_d155(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  size_t size;
  char unused2;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, &w->lat, &w->lon,
		     TGPS_FORMAT_LONGWORD, &w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, &w->proximity,
		     TGPS_FORMAT_CHAR_ARRAY, w->name, 30,
		     TGPS_FORMAT_CHAR_ARRAY, w->city, 24,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_INT, &w->alt,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_CHAR, &unused2,
		     TGPS_FORMAT_BYTE, &w->class,
		     TGPS_FORMAT_INT, &w->smbl16,
		     TGPS_FORMAT_BYTE, &w->display,
		     TGPS_FORMAT_END);
  w->smbl_type = TGPS_GARMIN_SYMBOLS_FULL;
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process waypoint data packet */
int tgps_wpt_data_in(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  int ret;

  tgps_clear_waypoint(w);
  
  if (tgps_is_supported(conn, 'D', 100))
    ret = tgps_wpt_data_in_d100(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 101))
    ret = tgps_wpt_data_in_d101(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 102))
    ret = tgps_wpt_data_in_d102(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 103))
    ret = tgps_wpt_data_in_d103(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 104))
    ret = tgps_wpt_data_in_d104(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 105))
    ret = tgps_wpt_data_in_d105(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 106))
    ret = tgps_wpt_data_in_d106(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 107))
    ret = tgps_wpt_data_in_d107(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 108))
    ret = tgps_wpt_data_in_d108(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 109))
    ret = tgps_wpt_data_in_d109(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 150))
    ret = tgps_wpt_data_in_d150(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 151))
    ret = tgps_wpt_data_in_d151(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 152))
    ret = tgps_wpt_data_in_d152(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 154))
    ret = tgps_wpt_data_in_d154(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 155))
    ret = tgps_wpt_data_in_d155(conn, packet, w);
  else if (packet->data_len == 60)
    ret = tgps_wpt_data_in_d100(conn, packet, w);
  else if (packet->data_len == 65)
    ret = tgps_wpt_data_in_d101(conn, packet, w);
  else if (packet->data_len == 66)
    ret = tgps_wpt_data_in_d102(conn, packet, w);
  else if (packet->data_len == 62)
    ret = tgps_wpt_data_in_d103(conn, packet, w);
  else if (packet->data_len == 67)
    ret = tgps_wpt_data_in_d104(conn, packet, w);
  else if (packet->data_len == 117)
    ret = tgps_wpt_data_in_d150(conn, packet, w);
  else if (packet->data_len == 126)
    ret = tgps_wpt_data_in_d151(conn, packet, w);
  else if (packet->data_len == 128)
    ret = tgps_wpt_data_in_d154(conn, packet, w);
  else if (packet->data_len == 129)
    ret = tgps_wpt_data_in_d155(conn, packet, w);
  else
    {
      fprintf(stderr, "Unknown waypoint packet type and length\n");
      return 0;
    }
  if (ret != 0)
    conn->packet_count--;
  return ret;
}

/* Process proximity waypoint data packet */
int tgps_prx_wpt_data_in(Tgps conn, TgpsPacket packet, TgpsWaypoint w)
{
  int ret;

  tgps_clear_waypoint(w);

  if (tgps_is_supported(conn, 'D', 400))
    ret = tgps_wpt_data_in_d400(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 403))
    ret = tgps_wpt_data_in_d403(conn, packet, w);
  else if (tgps_is_supported(conn, 'D', 450))
    ret = tgps_wpt_data_in_d450(conn, packet, w);
  else if (packet->data_len == 64)
    ret = tgps_wpt_data_in_d400(conn, packet, w);
  else if (packet->data_len == 66)
    ret = tgps_wpt_data_in_d403(conn, packet, w);
  else if (packet->data_len == 123)
    ret = tgps_wpt_data_in_d150(conn, packet, w);
  else
    {
      fprintf(stderr, "Unknown proximity waypoint packet type and length, trying to decode it as waypoint data\n");
      ret = tgps_wpt_data_in(conn, packet, w);
      return ret;
    }
  if (ret != 0)
    conn->packet_count--;
  return ret;
}

/* Send version 100 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d100(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_LONGWORD, w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 101 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d101(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_LONGWORD, w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, w->proximity,
		     TGPS_FORMAT_BYTE, w->smbl,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 102 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d102(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_LONGWORD, w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, w->proximity,
		     TGPS_FORMAT_INT, w->smbl16,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 103 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d103(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_LONGWORD, w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_BYTE, w->smbl,
		     TGPS_FORMAT_BYTE, w->display,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 104 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d104(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_LONGWORD, w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, w->proximity, 
		     TGPS_FORMAT_INT, w->smbl16,
		     TGPS_FORMAT_BYTE, w->display,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 105 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d105(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_INT, w->smbl16,
		     TGPS_FORMAT_STRING, w->wpt_ident,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 106 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d106(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_BYTE, w->class,
		     TGPS_FORMAT_BYTE_ARRAY, w->subclass, 13,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_INT, w->smbl16,
		     TGPS_FORMAT_STRING, w->wpt_ident,
		     TGPS_FORMAT_STRING, w->lnk_ident,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 107 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d107(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_LONGWORD, w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_BYTE, w->smbl,
		     TGPS_FORMAT_BYTE, w->display,
		     TGPS_FORMAT_FLOAT, w->proximity,
		     TGPS_FORMAT_BYTE, w->color,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 108 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d108(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;
  int color;

  color = w->color;
  if (color > 16)
    color -= 16;
  else if (color == 0)
    color = 255;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_BYTE, w->class,
		     TGPS_FORMAT_BYTE, color,
		     TGPS_FORMAT_BYTE, w->display,
		     TGPS_FORMAT_BYTE, w->attribute,
		     TGPS_FORMAT_INT, w->smbl16,
		     TGPS_FORMAT_BYTE_ARRAY, w->subclass, 18,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_FLOAT, w->altf,
		     TGPS_FORMAT_FLOAT, w->dpth,
		     TGPS_FORMAT_FLOAT, w->proximity,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_STRING, w->wpt_ident,
		     TGPS_FORMAT_STRING, w->comment_var,
		     TGPS_FORMAT_STRING, w->facility_var,
		     TGPS_FORMAT_STRING, w->city_var,
		     TGPS_FORMAT_STRING, w->addr_var,
		     TGPS_FORMAT_STRING, w->cross_road_var,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 109 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d109(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;
  int dspl_color;

  dspl_color = w->color;
  if (dspl_color > 16)
    dspl_color -= 16;
  else if (dspl_color == 0)
    dspl_color = 0x1f;
  dspl_color |= (w->display << 5);

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_BYTE, 0x01, /* Dtype */
		     TGPS_FORMAT_BYTE, w->class,
		     TGPS_FORMAT_BYTE, dspl_color,
		     TGPS_FORMAT_BYTE, w->attribute == 0 ? 0x70 : w->attribute,
		     TGPS_FORMAT_INT, w->smbl16,
		     TGPS_FORMAT_BYTE_ARRAY, w->subclass, 18,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_FLOAT, w->altf,
		     TGPS_FORMAT_FLOAT, w->dpth,
		     TGPS_FORMAT_FLOAT, w->proximity,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_LONGWORD, w->ete,
		     TGPS_FORMAT_STRING, w->wpt_ident,
		     TGPS_FORMAT_STRING, w->comment_var,
		     TGPS_FORMAT_STRING, w->facility_var,
		     TGPS_FORMAT_STRING, w->city_var,
		     TGPS_FORMAT_STRING, w->addr_var,
		     TGPS_FORMAT_STRING, w->cross_road_var,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 150 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d150(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_BYTE, w->class,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_INT, w->alt,
		     TGPS_FORMAT_CHAR_ARRAY, w->city, 24,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_CHAR_ARRAY, w->name, 30,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 151 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d151(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_LONGWORD, w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, w->proximity,
		     TGPS_FORMAT_CHAR_ARRAY, w->name, 30,
		     TGPS_FORMAT_CHAR_ARRAY, w->city, 24,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_INT, w->alt,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_CHAR, 0,
		     TGPS_FORMAT_BYTE, w->class, 
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 152 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d152(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_LONGWORD, w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, w->proximity,
		     TGPS_FORMAT_CHAR_ARRAY, w->name, 30,
		     TGPS_FORMAT_CHAR_ARRAY, w->city, 24,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_INT, w->alt,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_CHAR, 0,
		     TGPS_FORMAT_BYTE, w->class, 
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 154 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d154(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_LONGWORD, w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, w->proximity,
		     TGPS_FORMAT_CHAR_ARRAY, w->name, 30,
		     TGPS_FORMAT_CHAR_ARRAY, w->city, 24,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_INT, w->alt,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_CHAR, 0,
		     TGPS_FORMAT_BYTE, w->class,
		     TGPS_FORMAT_INT, w->smbl16,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 155 waypoint data packet */
TgpsPacket tgps_wpt_data_out_d155(Tgps conn, TgpsWaypoint w)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE,
		     TGPS_UPLOAD_DATA(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR_ARRAY, w->ident, 6,
		     TGPS_FORMAT_SEMICIRCLE, w->lat, w->lon,
		     TGPS_FORMAT_LONGWORD, w->unused_creation_time,
		     TGPS_FORMAT_CHAR_ARRAY, w->comment, 40,
		     TGPS_FORMAT_FLOAT, w->proximity,
		     TGPS_FORMAT_CHAR_ARRAY, w->name, 30,
		     TGPS_FORMAT_CHAR_ARRAY, w->city, 24,
		     TGPS_FORMAT_CHAR_ARRAY, w->state, 2,
		     TGPS_FORMAT_INT, w->alt,
		     TGPS_FORMAT_CHAR_ARRAY, w->cc, 2,
		     TGPS_FORMAT_CHAR, 0,
		     TGPS_FORMAT_BYTE, w->class,
		     TGPS_FORMAT_INT, w->smbl16,
		     TGPS_FORMAT_BYTE, w->display,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send waypoint data packet */
TgpsPacket tgps_wpt_data_out(Tgps conn, TgpsWaypoint w)
{
  if (tgps_is_supported(conn, 'D', 100))
    return tgps_wpt_data_out_d100(conn, w);
  else if (tgps_is_supported(conn, 'D', 101))
    return tgps_wpt_data_out_d101(conn, w);
  else if (tgps_is_supported(conn, 'D', 102))
    return tgps_wpt_data_out_d102(conn, w);
  else if (tgps_is_supported(conn, 'D', 103))
    return tgps_wpt_data_out_d103(conn, w);
  else if (tgps_is_supported(conn, 'D', 104))
    return tgps_wpt_data_out_d104(conn, w);
  else if (tgps_is_supported(conn, 'D', 105))
    return tgps_wpt_data_out_d105(conn, w);
  else if (tgps_is_supported(conn, 'D', 106))
    return tgps_wpt_data_out_d106(conn, w);
  else if (tgps_is_supported(conn, 'D', 107))
    return tgps_wpt_data_out_d107(conn, w);
  else if (tgps_is_supported(conn, 'D', 108))
    return tgps_wpt_data_out_d108(conn, w);
  else if (tgps_is_supported(conn, 'D', 109))
    return tgps_wpt_data_out_d109(conn, w);
  else if (tgps_is_supported(conn, 'D', 150))
    return tgps_wpt_data_out_d150(conn, w);
  else if (tgps_is_supported(conn, 'D', 151))
    return tgps_wpt_data_out_d151(conn, w);
  else if (tgps_is_supported(conn, 'D', 152))
    return tgps_wpt_data_out_d152(conn, w);
  else if (tgps_is_supported(conn, 'D', 154))
    return tgps_wpt_data_out_d154(conn, w);
  else if (tgps_is_supported(conn, 'D', 155))
    return tgps_wpt_data_out_d155(conn, w);
  else
    {
      fprintf(stderr, "Unknown waypoint protocol format\n");
      return NULL;
    }
}
