/*
 * tgps.h -- GPS protocol definitions.
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/tgps.h,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 15:11 Apr  7 2000 kivinen
 *	  Last Modification : 00:51 Jan  8 2004 kivinen
 *	  Last check in     : $Date: 2003/08/18 20:12:54 $
 *	  Revision number   : $Revision: 1.15 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.293
 *	  Edit time	    : 133 min
 *
 *	  Description       : GPS protocol definitions
 *
 *	  $Log: tgps.h,v $
 *	  Revision 1.15  2003/08/18 20:12:54  kivinen
 *	  	Added -C option (ignore errors), use 1 byte ack/nak messages
 *	  	for most devices.
 *
 *	  Revision 1.14  2001/08/23 15:43:25  kivinen
 *	  	Added extra fields to almanac ad track records.
 *
 *	  Revision 1.13  2000/10/02 15:43:51  kivinen
 *	  	Changed to use tgps_print.
 *
 *	  Revision 1.12  2000/08/05 00:44:41  kivinen
 *	  	Removed warnings.
 *
 *	  Revision 1.11  2000/07/26 17:09:57  kivinen
 *	  	Added some prototypes.
 *
 *	  Revision 1.10  2000/07/17 21:34:22  kivinen
 *	  	Added retry count and linux serial fix kludge.
 *
 *	  Revision 1.9  2000/07/12 22:20:26  kivinen
 *	  	Final upload support.
 *
 *	  Revision 1.8  2000/07/08 21:17:00  kivinen
 *	  	Fixed bug in duplicate packets being acked.
 *
 *	  Revision 1.7  2000/07/08 20:43:48  kivinen
 *	  	Added -v option.
 *
 *	  Revision 1.6  2000/07/08 17:38:51  kivinen
 *	  	Added time.h.
 *
 *	  Revision 1.5  2000/07/07 20:41:16  kivinen
 *	  	Added support for ignoring compressed tracks. Added linux
 *	  	support.
 *
 *	  Revision 1.4  2000/07/06 23:07:23  kivinen
 *	  	Added initial upload support.
 *
 *	  Revision 1.3  2000/04/30 01:33:50  kivinen
 *	  	Changed invalid float detection code.
 *
 *	  Revision 1.2  2000/04/30 01:03:48  kivinen
 *	  	Updated to rev 03 document.
 *
 *	  Revision 1.1  2000/04/29 16:40:40  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#ifndef TGPS_H
#define TGPS_H

#include "tgpsconfig.h"

#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif /* HAVE_SYS_TIME_H */
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif /* HAVE_SYS_TYPES_H */
#ifdef STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#endif /* STDC_HEADERS */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#ifdef HAVE_TERMIOS_H
#include <termios.h>
#endif /* HAVE_TERMIOS_H */
#include <math.h>
#ifndef M_PI
#define	M_PI		3.14159265358979323846	/* pi */
#endif /* M_PI */

#define WGS84_EARTH_RADIUS_MAJOR	6378137.0
#define WGS84_INVF			298.257223563
#define WGS84_EARTH_RADIUS_MINOR	\
	(WGS84_EARTH_RADIUS_MAJOR * (1 - (1 / WGS84_INVF)))
#define WGS84_EARTH_RADIUS_MAJOR2	\
	(WGS84_EARTH_RADIUS_MAJOR * WGS84_EARTH_RADIUS_MAJOR)
#define WGS84_EARTH_RADIUS_MINOR2	\
	(WGS84_EARTH_RADIUS_MINOR * WGS84_EARTH_RADIUS_MINOR)

#define WGS84_EARTH_RADIUS(lat)		\
	(1 / (sqrt((sin(lat) * sin(lat) / WGS84_EARTH_RADIUS_MAJOR2) + \
		   (cos(lat) * cos(lat) / WGS84_EARTH_RADIUS_MINOR2))))
#define DEG2RAD(deg) (M_PI * (deg) / 180.0)

#if SIZEOF_SHORT == 2
typedef short tgps_int16_t;
typedef unsigned short tgps_uint16_t;
#elif SIZEOF_INT == 2
typedef int tgps_int16_t;
typedef unsigned int tgps_uint16_t;
#elif SIZEOF_LONG == 2
typedef long tgps_int16_t;
typedef unsigned long tgps_uint16_t;
#endif

#if SIZEOF_INT == 4
typedef int tgps_int32_t;
typedef unsigned int tgps_uint32_t;
#elif SIZEOF_SHORT == 4
typedef short tgps_int32_t;
typedef unsigned short tgps_uint32_t;
#elif SIZEOF_LONG == 4
typedef long tgps_int32_t;
typedef unsigned long tgps_uint32_t;
#endif

#define BUFLEN			1024
#define TGPS_RETRY_LIMIT	60
#define TGPS_PVT_RETRY		60
#ifdef __linux__
#define LINUX_SERIAL_FIX
#endif

/* Invalid values */
#define TGPS_INVALID_CHAR	-127
#define TGPS_INVALID_INT	-32767
#define TGPS_INVALID_LONG	-2147483647L
#define TGPS_INVALID_FLOAT	-1.0e25
#define TGPS_INVALID_FLOAT_TEST	-1.0e24

/* Garmin GPS base time */
#define TGPS_BASE_TIME 631065600

typedef enum {
  TGPS_DATA_VERSION_GPSTRANS = 0,
  TGPS_DATA_VERSION_TGPS_V1 = 1
} TgpsDataVersion;

typedef enum {
  TGPS_DEG_NONE,
  TGPS_DEG_MIN_SEC,
  TGPS_DEG_MIN,
  TGPS_DEG
} TgpsDegreeFormat; 

typedef enum {
  TGPS_OPERATION_NONE,
  TGPS_DOWNLOAD_WAYPOINTS,
  TGPS_DOWNLOAD_ROUTES,
  TGPS_DOWNLOAD_TRACK,
  TGPS_DOWNLOAD_ALMANAC,
  TGPS_DOWNLOAD_TIME,
  TGPS_DOWNLOAD_POSITION,
  TGPS_DOWNLOAD_PROXIMITY,
  TGPS_SET_TIME,
  TGPS_UPLOAD,
  TGPS_START_PVT
} TgpsOperation;

typedef struct TgpsPacketRec {
  struct TgpsPacketRec *next;	/* Next packet in free list */
  unsigned char *data;		/* Packet data */
  size_t data_len;		/* Length of the packet */
  size_t alloc_size;		/* Allocated length of the data */
  int retry_cnt;		/* Retry count */
} *TgpsPacket;

typedef struct TgpsRec {
  int tty;
  struct termios orig_tty;

  int product_id;		/* Garmin product id */
  int software_version;		/* Garmin software version */
  char *product_description;	/* Garmin product description */
  int *phys_prot_ids;		/* Physical protocol ids */
  int phys_prot_id_cnt;		/* Count of physical protocol ids */
  int *link_prot_ids;		/* Link protocol ids */
  int link_prot_id_cnt;		/* Count of Link protocol ids */
  int *appl_prot_ids;		/* Application protocol ids */
  int appl_prot_id_cnt;		/* Count of Application protocol ids */
  int *data_prot_ids;		/* Data protocol ids */
  int data_prot_id_cnt;		/* Count of Data protocol ids */

  unsigned char *buffer_in;	/* Input buffer */
  size_t buffer_in_alloc_size;	/* Allocated size of input buffer */
  size_t buffer_in_start;	/* First used byte */
  size_t buffer_in_len;		/* Bytes in input buffer */

  unsigned char *buffer_out;	/* Output buffer */
  size_t buffer_out_alloc_size;	/* Allocated size of output buffer */
  size_t buffer_out_start;	/* First used byte */
  size_t buffer_out_len;	/* Bytes in output buffer */

  unsigned char *buffer_last_packet; /* Last packet received */
  size_t buffer_last_packet_len; /* Length of last packet received */

  TgpsPacket free_list;		/* Pointer to first free packet */

  TgpsPacket command[256];	/* Queue of outgoing packets for each command
				   type */
  int packet_count;		/* Number of packets to receive still */
  int packet_count_total;	/* Total number of packets to receive */
  int xfer_done;		/* Is the transfer done */
  int pvt_active;		/* Is pvt active */
  TgpsDataVersion data_version; /* Display format version */
  TgpsDegreeFormat degree_format; /* Format of degree output */
  FILE *output_stream;		/* Output file stream */
  FILE *input_stream;		/* Input file stream */
  char *input_buffer;		/* Input buffer */
  size_t input_buffer_len;	/* Length of the data in input buffer */
  size_t input_buffer_alloc;	/* Alloc length of the input buffer */
  TgpsOperation operation;	/* Operation to do */
  int upload;			/* Upload */
#define TGPS_UPLOAD_COMMAND		0x00ff
#define TGPS_UPLOAD_ACTIVATED		0x0100
#define TGPS_UPLOAD_WAITING_FOR_ACK	0x0200
  int verbose;			/* Verbose mode */
  int flags;			/* Flags */
#define TGPS_FLAGS_IGNORE_COMPRESSED_TRACKS	0x0001
#define TGPS_FLAGS_DO_NOT_EXIT			0x0002
#define TGPS_FLAGS_IGNORE_ERRORS		0x0004
#define TGPS_FLAGS_USE_1_BYTE_ACK		0x0008
} *Tgps;

/* Put packet back to free list */
void tgps_free_packet(Tgps conn, TgpsPacket packet);

/* Get packet from the free list */
TgpsPacket tgps_get_packet(Tgps conn, size_t size);

/* Check if the protocol version is supported */
int tgps_is_supported(Tgps conn, unsigned char type, int version);

#include "garmin.h"

typedef enum {
  TGPS_FORMAT_CHAR,		/* char */
  TGPS_FORMAT_BYTE,		/* unsigned char */
  TGPS_FORMAT_INT,		/* int */
  TGPS_FORMAT_WORD,		/* unsigned int */
  TGPS_FORMAT_LONG,		/* long */
  TGPS_FORMAT_LONGWORD,		/* unsigned long */
  TGPS_FORMAT_FLOAT,		/* float */
  TGPS_FORMAT_DOUBLE,		/* double */
  TGPS_FORMAT_CHAR_ARRAY,	/* unsigned char *, size_t */
  TGPS_FORMAT_BYTE_ARRAY,	/* unsigned char *, size_t (not NUL terminated) */
  TGPS_FORMAT_STRING,		/* unsigned char * */
  TGPS_FORMAT_BOOLEAN,		/* int */
  TGPS_FORMAT_SEMICIRCLE,	/* long, long */
  TGPS_FORMAT_RADIAN,		/* double, double */
  TGPS_FORMAT_LEN,		/* none, encodes here a full packet length as
				   a one byte. */
  TGPS_FORMAT_END
} TgpsFormat;

/* Waypoint structure */
typedef struct TgpsWaypointRec {
  char ident[7];		/* Name of the waypoint */
  long lat;			/* Position */
  long lon;
  unsigned long unused_creation_time; /* Creation time or zero for newer gps,
					 ignored when uploaded */
  char comment[41];		/* Comment field */
  TgpsGarminSymbolType smbl_type; /* Type of symbols, 16 or full 65535 */
  char smbl;			/* 8 bit symbol field */
  int smbl16;			/* 16 bit symbol field */
  char display;			/* Display type */
  char color;			/* Colors */
  char attribute;		/* Attribute */
  float proximity;		/* Proximity distance */
  char class;			/* Class */
  char subclass[18];		/* Subclass 12 or 18 chars */
  char cc[3];			/* Country */
  char city[25];		/* City */
  char state[3];		/* State */
  char name[31];		/* Name */
  int idx;			/* proximity index */
  int alt;			/* Altitude as int */
  float altf;			/* Altitude as float */
  float dpth;			/* Depth */
  unsigned long ete;		/* Estimated time en route in seconds */
  char *wpt_ident;		/* Null terminated version of ident */
  char *lnk_ident;		/* Null terminated version of link ident */
  char *comment_var;		/* Null terminated version of comment */
  char *facility_var;		/* Null terminated */
  char *city_var;		/* Null terminated */
  char *addr_var;		/* Null terminated */
  char *cross_road_var;		/* Null terminated */
} *TgpsWaypoint, TgpsWaypointStruct;

/* Route header structure */
typedef struct TgpsRouteHeaderRec {
  char number;
  char comment[21];
  char *rte_ident;		/* Null terminated version of ident */
} *TgpsRouteHeader, TgpsRouteHeaderStruct;

/* Route link structure */
typedef struct TgpsRouteLnkRec {
  char class;
  char subclass[18];
  char *ident;			/* Null terminated ident */
} *TgpsRouteLnk, TgpsRouteLnkStruct;

/* Track point structure */
typedef struct TgpsTrackRec {
  long lat;
  long lon;
  unsigned long time;
  int new_track;
  float alt;
  float dpth;
  char extra[3];		/* Extra data */
} *TgpsTrack, TgpsTrackStruct;

/* Track header structure */
typedef struct TgpsTrackHeaderRec {
  int display; 
  char color;
  char *ident;
  unsigned long line_width;
  int fill;
} *TgpsTrackHeader, TgpsTrackHeaderStruct;

/* Almanac structure */
typedef struct TgpsAlmanacRec {
  char svid;
  int wn;
  float toa;
  float af0;
  float af1;
  float e;
  float sqrta;
  float m0;
  float w;
  float omg0;
  float odot;
  float i;
  char hlth;
  char extra[3];		/* Extra data */
} *TgpsAlmanac, TgpsAlmanacStruct;

/* Time structure */
typedef struct TgpsTimeRec {
  unsigned char month; 
  unsigned char day; 
  unsigned int year; 
  int hour;
  unsigned char minute;
  unsigned char second;
} *TgpsTime, TgpsTimeStruct;

/* Position structure */
typedef struct TgpsPositionRec {
  double lat;
  double lon;
} *TgpsPosition, TgpsPositionStruct;

/* PVT structure */
typedef struct TgpsPvtRec {
  float alt;
  float epe;
  float eph;
  float epv;
  int fix;
  double tow;
  double lat;
  double lon;
  float east;
  float north;
  float up;
  float msl_hght;
  int leap_scnds;
  long wn_days;
} *TgpsPvt, TgpsPvtStruct;

/* Convert semicircle to degrees */
double tgps_semicircle_to_deg(long semicircle);

/* Convert degrees to semicircle */
long tgps_deg_to_semicircle(double degrees);

/* Convert radians to degrees */
double tgps_radians_to_deg(double radians);

/* Convert degrees to radians */
double tgps_deg_to_radians(double degrees);

/* Is character array empty */
int tgps_is_empty(char *str);

/* Print buffer. */
void tgps_print_buffer(char *txt, unsigned char *buffer, size_t buflen);

/* Encode data to the buffer using varargs list. Return number of bytes written
   to buffer, or -1 if the data didn't fit to the buffer */
size_t tgps_encode_va(unsigned char *buffer, size_t buffer_len, va_list ap);

/* Encode data to the buffer using varargs list. Return number of bytes written
   to buffer, or -1 if the data didn't fit to the buffer */
size_t tgps_encode(unsigned char *buffer, size_t buffer_len, ...);

/* Decode data from the buffer using varargs list. Return number of bytes
   read from buffer, or -1 if there was not enough data in buffer */
size_t tgps_decode_va(unsigned char *buffer, size_t buffer_len, va_list ap);

/* Decode data from the buffer using varargs list. Return number of bytes
   read from buffer, or -1 if there was not enough data in buffer */
size_t tgps_decode(unsigned char *buffer, size_t buffer_len, ...);

/* Write data to output file */
void tgps_print(Tgps conn, const char *format, ...);

/* Get line from input file, and put it to input buffer. Returns true if
   successfull, and false if end of file. */
int tgps_gets(Tgps conn);

#endif /* TGPS_H */
