/*
 * t2ps.c -- Print track to postscript printer
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/t2ps.c,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 09:24 Jul 21 2000 kivinen
 *	  Last Modification : 01:56 Jan  8 2004 kivinen
 *	  Last check in     : $Date: 2002/01/30 21:29:43 $
 *	  Revision number   : $Revision: 1.5 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.780
 *	  Edit time	    : 376 min
 *
 *	  Description       : Print track to postscript printer
 *
 *	  $Log: t2ps.c,v $
 *	  Revision 1.5  2002/01/30 21:29:43  kivinen
 *	  	Added speed color support.
 *
 *	  Revision 1.4  2000/10/02 15:42:51  kivinen
 *	  	Added colors for speeds.
 *
 *	  Revision 1.3  2000/08/05 00:44:22  kivinen
 *	  	Updated to new interface.
 *
 *	  Revision 1.2  2000/07/23 20:24:15  kivinen
 *	  	Added pvt and waypoint support.
 *
 *	  Revision 1.1  2000/07/21 22:18:43  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#include "tgps.h"
#include "data.h"
#include "packet.h"
#include "ieee-double.h"
#include "input.h"
#include "map.h"

typedef enum {
  TGPS_TEXT_DIRECTION_RIGHT = 0,
  TGPS_TEXT_DIRECTION_RIGHT_JUSTIFIED = 1,
  TGPS_TEXT_DIRECTION_UP = 2,
  TGPS_TEXT_DIRECTION_UP_JUSTIFIED = 3,
  TGPS_TEXT_DIRECTION_LEFT = 4,
  TGPS_TEXT_DIRECTION_LEFT_JUSTIFIED = 5,
  TGPS_TEXT_DIRECTION_DOWN = 6,
  TGPS_TEXT_DIRECTION_DOWN_JUSTIFIED = 7,
} TgpsTextDirection;

Tgps tgps_open(char *output_file, char *input_file);
int tgps_close(Tgps conn);
double degree_in_meters_longitude(double lat);
double degree_in_meters_latitude(double lat);
void tgps_ps_print_header(Tgps conn);
void tgps_ps_print_footer(Tgps conn);
double tgps_ps_latitude(double lat, double lon);
double tgps_ps_longitude(double lat, double lon);
int tgps_ps_draw_line(Tgps conn, double lat1, double lon1,
		      double lat2, double lon2);
void tgps_ps_draw_speed(Tgps conn, unsigned long t, unsigned long pt,
			double lat1, double lon1, 
			double lat2, double lon2);
void tgps_ps_draw_time(Tgps conn, unsigned long t, unsigned long pt,
		       double lat, double lon);
void tgps_set_attrs(Tgps conn, unsigned long t, double lat, double lon);
void tgps_ps_draw_line_start(Tgps conn, unsigned long t,
			    double lat, double lon);
void tgps_ps_draw_line_point(Tgps conn, unsigned long t,
			     double lat, double lon);
void tgps_ps_draw_line_end(Tgps conn);
void tgps_point(Tgps conn, int new_track, unsigned long t,
		double lat, double lon, unsigned long time_limit);
void tgps_store_point(Tgps conn, unsigned long t,
		      double lat, double lon);
void tgps_ps_draw_circle(Tgps conn, double lat, double lon,
			 double size);
void tgps_ps_draw_text(Tgps conn, double lat, double lon,
		       int tx, int ty, double size,
		       TgpsTextDirection direction, const char *text);
void tgps_print_axis(Tgps conn);
void tgps_set_clip(Tgps conn);
int tgps_is_visible(double lat, double lon);

/* Open connection to gps */
Tgps tgps_open(char *output_file, char *input_file)
{
  Tgps conn;

  conn = calloc(1, sizeof(*conn));
  if (conn == NULL)
    return NULL;

  if (output_file == NULL || strcmp(output_file, "-") == 0)
    conn->output_stream = stdout;
  else
    {
#ifdef HAVE_STRFTIME
      char buffer[512];
      struct tm *tptr;
      time_t t;

      t = time(NULL);
      tptr = localtime(&t);
      strftime(buffer, sizeof(buffer), output_file, tptr);
      conn->output_stream = fopen(buffer, "wb");
#else /* HAVE_STRFTIME */
      conn->output_stream = fopen(output_file, "wb");
#endif /* HAVE_STRFTIME */
    }

  if (conn->output_stream == NULL)
    {
      perror("Could not open output file");
      free(conn);
      return NULL;
    }

  if (input_file == NULL || strcmp(input_file, "-") == 0)
    conn->input_stream = stdin;
  else
    conn->input_stream = fopen(input_file, "rb");
  if (conn->input_stream == NULL)
    {
      perror("Could not open input file");
      if (conn->output_stream != stdout)
	fclose(conn->output_stream);
      free(conn);
      return NULL;
    }

  conn->product_id = 0;
  conn->software_version = 0;
  conn->phys_prot_ids = NULL;
  conn->phys_prot_id_cnt = 0;
  conn->link_prot_ids = NULL;
  conn->link_prot_id_cnt = 0;
  conn->appl_prot_ids = NULL;
  conn->appl_prot_id_cnt = 0;
  conn->data_prot_ids = NULL;
  conn->data_prot_id_cnt = 0;
  conn->packet_count = 0;
  conn->packet_count_total = 0;
  conn->xfer_done = 0;
  conn->degree_format = TGPS_DEG;
  conn->data_version = TGPS_DATA_VERSION_TGPS_V1;
  conn->pvt_active = 0;

  conn->input_buffer_len = -1;
  conn->input_buffer_alloc = 1024;
  conn->input_buffer = malloc(conn->input_buffer_alloc);
  if (conn->input_buffer == NULL)
    {
      if (conn->output_stream != stdout)
	fclose(conn->output_stream);
      if (conn->input_stream != stdin)
	fclose(conn->input_stream);
      free(conn);
      return NULL;
    }

  return conn;
}

/* Close connection to gps */
int tgps_close(Tgps conn)
{
  if (conn->output_stream != stdout)
    fclose(conn->output_stream);
  if (conn->input_stream != stdin)
    fclose(conn->input_stream);
  free(conn->product_description);
  free(conn->input_buffer);
  free(conn->phys_prot_ids);
  free(conn->link_prot_ids);
  free(conn->appl_prot_ids);
  free(conn->data_prot_ids);
  free(conn);
  return 0;
}

/* Check if the protocol version is supported */
int tgps_is_supported(Tgps conn, unsigned char type, int version)
{
  return 0;
}

int xsize = 801;
int ysize = 401;
int grid = 36;
double scale = 50000.0;
double latitude = 0.0;
double longitude = 0.0;
double fontsize = 8.0;
double textoffsetx = 3.0;
double textoffsety = -2.0;
double waypointcircle = 1.0;
double trackpointcircle = 0.1;

double latitude_max, latitude_min;
double longitude_max, longitude_min;

#define NUMBER_OF_SPEEDS	256

int bw_flag = 0;
int t_flag = 0;

unsigned long current_color = 0;
unsigned long next_color = 0;
unsigned long current_width = 0;
unsigned long next_width = 0;
int filling = 0;

double speed_limits[NUMBER_OF_SPEEDS] = {
  0.001, 27, 54, -1
};
unsigned long speed_colors[NUMBER_OF_SPEEDS] = {
  0x000000, 0x00ff00, 0xffff00, 0xff0000
};

unsigned long garmin_colors[] = {
  0x000000,		/* 0, 16, Black */
  0x800000,		/* 1, 17, Dark_Red */
  0x008000,		/* 2, 18, Dark_Green */
  0xc0c000,		/* 3, 19, Dark_Yellow */
  0x000080,		/* 4, 20, Dark_Blue */
  0x800080,		/* 5, 21, Dark_Magenta */
  0x008080,		/* 6, 22, Dark_Cyan */
  0xc0c0c0,		/* 7, 23, Light_Gray */
  0x808080,		/* 8, 24, Dark_Gray */
  0xff0000,		/* 9, 25, Red */
  0x00ff00,		/* 10, 26, Green */
  0xffff00,		/* 11, 27, Yellow */
  0x0000ff,		/* 12, 28, Blue */
  0xff00ff,		/* 13, 29, Magenta */
  0x00ffff,		/* 14, 30, Cyan */
  0xffffff		/* 15, 31, White */
};

unsigned long time_colors[] = {
  0xff0000, 0xff8000, 0xffff00, 0x80ff00,
  0x00ff00, 0x00ff80, 0x00ffff, 0x0080ff,
  0x0000ff, 0x8000ff, 0xff00ff, 0xff0080
};

int time_colors_count = sizeof(time_colors) / sizeof(time_colors[0]);

int t_limit = 3600;

/* Return one degree in latitude in meters at given latitude */
double degree_in_meters_latitude(double lat)
{
  return 2 * M_PI * WGS84_EARTH_RADIUS(DEG2RAD(lat)) / 360;
}

/* Return one degree in longitude in meters at given latitude */
double degree_in_meters_longitude(double lat)
{
  return degree_in_meters_latitude(lat) * cos(DEG2RAD(lat));
}

/* Print postscript header */
void tgps_ps_print_header(Tgps conn)
{
  time_t t;
  t = time(NULL);

  fprintf(conn->output_stream, "%%!PS-Adobe-3.0\n");
  fprintf(conn->output_stream, "%%%%Creator: t2ps\n");
  fprintf(conn->output_stream, "%%%%Size: %d %d\n", xsize, ysize);
  fprintf(conn->output_stream, "%%%%Center: %g %g\n", longitude, latitude);
  fprintf(conn->output_stream, "%%%%Scale: %g\n", scale);
  fprintf(conn->output_stream, "%%%%CreationDate: %s", ctime(&t));
  fprintf(conn->output_stream,
	  "%%%%DocumentNeededResources: font HelveticaNarrow\n");
  fprintf(conn->output_stream, "%%%%Pages: 1\n");
  fprintf(conn->output_stream, "%%%%PageOrder: Ascend\n");
  fprintf(conn->output_stream, "%%%%BoundingBox: 30 30 %d %d\n",
	  xsize + 80, ysize + 80);
  fprintf(conn->output_stream, "%%%%EndoComments\n");
  fprintf(conn->output_stream, "%%%%BeginProlog\n");
  fprintf(conn->output_stream, "%%%%IncludeResource: font HelveticaNarrow\n");
  fprintf(conn->output_stream, "/l { newpath moveto lineto stroke } def\n");
  fprintf(conn->output_stream, "/ls { newpath moveto } def\n");
  fprintf(conn->output_stream, "/lp { lineto } def\n");
  fprintf(conn->output_stream, "/le { stroke } def\n");
  fprintf(conn->output_stream, "/fe { closepath fill } def\n");
  fprintf(conn->output_stream,
	  "/c { newpath 0 360 arc closepath fill } def\n");
  fprintf(conn->output_stream, "/s { setrgbcolor } def\n");
  fprintf(conn->output_stream, "/t { 0 0 0 s show } def\n");
  fprintf(conn->output_stream, "/tj { 0 0 0 s dup stringwidth pop neg 0 rmoveto show } def\n");
  fprintf(conn->output_stream,
	  "/f { 0 0 0 s font exch scalefont setfont } def\n");
  fprintf(conn->output_stream, "/rs { gsave translate rotate } def\n");
  fprintf(conn->output_stream, "/re { grestore } def\n");
  fprintf(conn->output_stream, "%%%%EndProlog\n");
  fprintf(conn->output_stream, "%%%%Page: 1 1\n");
  fprintf(conn->output_stream, "%%%%BeginPageSetup\n");
  fprintf(conn->output_stream, "/font /Helvetica-Narrow findfont def\n");
  fprintf(conn->output_stream, "50 50 translate 0 setlinewidth\n");
  fprintf(conn->output_stream, "%%%%EndPageSetup\n");
}

/* Print postscript footer */
void tgps_ps_print_footer(Tgps conn)
{
  fprintf(conn->output_stream, "showpage\n%%%%Trailer\n%%%%EOF\n");
}

/* Convert latitude to postscript coordinates */
double tgps_ps_latitude(double lat, double lon)
{
  return (ysize / 2) + (lat - latitude) * degree_in_meters_latitude(lat)
    / scale;
}

/* Convert longitude to postscript coordinates */
double tgps_ps_longitude(double lat, double lon)
{
  return (xsize / 2) + (lon - longitude) * degree_in_meters_longitude(lat) /
    scale;
}

/* Send color by speed */
void tgps_ps_draw_speed(Tgps conn, unsigned long t, unsigned long pt,
			double lat1, double lon1, 
			double lat2, double lon2)
{
  double dist, speed;
  
  int i;

  if (bw_flag)
    return;
  dist = tgps_point_distance(DEG2RAD(lat1),
			     DEG2RAD(lon1),
			     DEG2RAD(lat2),
			     DEG2RAD(lon2));
  speed = dist / ((double) t - pt);
  
  for(i = 0; i < NUMBER_OF_SPEEDS; i++)
    {
      if (speed_limits[i] < 0.0)
	break;
      if (speed < speed_limits[i])
	{
	  next_color = speed_colors[i];
	  return;
	}
    }
  next_color = 0x000000;
  return;
}

/* Send color by time */
void tgps_ps_draw_time(Tgps conn, unsigned long t, unsigned long pt,
		       double lat, double lon)
{
  static int color_index = -1;
  char buffer[10];
  
  if (!t_flag)
    return;
  if (color_index != -1 && (int) (t / 3600) == (int) (pt / 3600))
    return;

  color_index++;
  if (color_index == time_colors_count)
    color_index = 0;
  fprintf(conn->output_stream, "gsave 0 0 0 s\n");
  snprintf(buffer, sizeof(buffer), "%d:00", (int) ((t / 3600) % 24));
  tgps_ps_draw_text(conn, lat, lon, 1, 1, 7,
		    TGPS_TEXT_DIRECTION_RIGHT, buffer);
  fprintf(conn->output_stream, "grestore\n");
  next_color = time_colors[color_index];
  return;
}

double prev_nlat, prev_nlon;
double prev_lat, prev_lon;
unsigned long prev_time = 0;
/* Number of points drawn to current line */
int draw_line = 0;

/* Set color / width based on speed / time / given setting. */
void tgps_set_attrs(Tgps conn, unsigned long t, double lat, double lon)
{
  tgps_ps_draw_time(conn, t, prev_time, lat, lon);
  tgps_ps_draw_speed(conn, t, prev_time, lat, lon, prev_lat, prev_lon);

  if (next_color != current_color || next_width != current_width)
    {
      if (draw_line > 1)
	tgps_ps_draw_line_end(conn);
      if (next_color != current_color)
	{
	  current_color = next_color;
	  fprintf(conn->output_stream, "%g %g %g s\n",
		  (double) ((current_color >> 16) & 0xff) / 255,
		  (double) ((current_color >> 8) & 0xff) / 255,
		  (double) (current_color & 0xff) / 255);
	}
      if (next_width != current_width)
	{
	  current_width = next_width;
	  fprintf(conn->output_stream, "%ld setlinewidth\n",
		  current_width / 10);
	}
    }
}

/* Store point for prev point */
void tgps_store_point(Tgps conn, unsigned long t,
		      double lat, double lon)
{
  double nlat, nlon;

  nlat = tgps_ps_latitude(lat, lon);
  nlon = tgps_ps_longitude(lat, lon);

  prev_nlat = nlat;
  prev_nlon = nlon;
  prev_lat = lat;
  prev_lon = lon;
  prev_time = t;
}

/* Start line in postscript file */
void tgps_ps_draw_line_start(Tgps conn, unsigned long t,
			     double lat, double lon)
{
  if (draw_line)
    tgps_ps_draw_line_end(conn);
  tgps_store_point(conn, t, lat, lon);

  fprintf(conn->output_stream, "%.1f %.1f ls\n", prev_nlon, prev_nlat);
  draw_line = 1;
}

/* Point in line in postscript file */
void tgps_ps_draw_line_point(Tgps conn, unsigned long t,
			     double lat, double lon)
{
  double nlat, nlon;

  nlat = tgps_ps_latitude(lat, lon);
  nlon = tgps_ps_longitude(lat, lon);

  if ((int) (nlat * 10 + 0.5) !=
      (int) (prev_nlat * 10 + 0.5) ||
      (int) (nlon * 10 + 0.5) !=
      (int) (prev_nlon * 10 + 0.5))
    {
      fprintf(conn->output_stream, "%.1f %.1f lp\n", nlon, nlat);
      prev_nlat = nlat;
      prev_nlon = nlon;
      prev_lat = lat;
      prev_lon = lon;
      prev_time = t;
      draw_line++;
    }
}

/* End line in postscript file */
void tgps_ps_draw_line_end(Tgps conn)
{
  if (draw_line)
    {
      if (filling)
	fprintf(conn->output_stream, "fe\n");
      else
	fprintf(conn->output_stream, "le\n");
    }
  draw_line = 0;
}

void tgps_point(Tgps conn, int new_track, unsigned long t,
		double lat, double lon, unsigned long time_limit)
{
  if (new_track || t - prev_time > time_limit)
    {
      tgps_ps_draw_line_start(conn, t, lat, lon);
    }
  else
    {
      tgps_set_attrs(conn, t, lat, lon);
      if (!draw_line)
	tgps_ps_draw_line_start(conn, prev_time, prev_lat, prev_lon);
      tgps_ps_draw_line_point(conn, t, lat, lon);
    }
}

/* Draw line in postscript file */
int tgps_ps_draw_line(Tgps conn, double lat1, double lon1,
		      double lat2, double lon2)
{
  double nlat1, nlat2, nlon1, nlon2;

  nlat1 = tgps_ps_latitude(lat1, lon1);
  nlon1 = tgps_ps_longitude(lat1, lon1);
  nlat2 = tgps_ps_latitude(lat2, lon2);
  nlon2 = tgps_ps_longitude(lat2, lon2);

  if ((int) (nlat1 * 10 + 0.5) !=
      (int) (nlat2 * 10 + 0.5) ||
      (int) (nlon1 * 10 + 0.5) !=
      (int) (nlon2 * 10 + 0.5))
    {
      fprintf(conn->output_stream, "%.1f %.1f %.1f %.1f l\n",
	      nlon2, nlat2, nlon1, nlat1);
      return 1;
    }
  return 0;
}

/* Draw circle in postscript file */
void tgps_ps_draw_circle(Tgps conn, double lat, double lon,
			 double size)
{
  fprintf(conn->output_stream,
	  "%.1f %.1f %.1f c\n",
	  tgps_ps_longitude(lat, lon), tgps_ps_latitude(lat, lon), size);
}

/* Draw text in postscript file */
void tgps_ps_draw_text(Tgps conn, double lat, double lon,
		       int tx, int ty, double size,
		       TgpsTextDirection direction, const char *text)
{
  static double current_size = -1;
  int angle;

  switch (direction)
    {
    case TGPS_TEXT_DIRECTION_RIGHT:
    case TGPS_TEXT_DIRECTION_RIGHT_JUSTIFIED:
      angle = 0;
      break;
    case TGPS_TEXT_DIRECTION_UP:
    case TGPS_TEXT_DIRECTION_UP_JUSTIFIED:
      angle = 90;
      break;
    case TGPS_TEXT_DIRECTION_LEFT:
    case TGPS_TEXT_DIRECTION_LEFT_JUSTIFIED:
      angle = 180;
      break;
    case TGPS_TEXT_DIRECTION_DOWN:
    case TGPS_TEXT_DIRECTION_DOWN_JUSTIFIED:
      angle = -90;
      break;
    }

  if (size != current_size)
    {
      fprintf(conn->output_stream, "%.1f f ", size);
      current_size = size;
    }
  fprintf(conn->output_stream, "%d %.1f %.1f rs 0 0 moveto (", angle, 
	  tgps_ps_longitude(lat, lon) + tx,
	  tgps_ps_latitude(lat, lon) + ty);

  while (isspace(*text))
    text++;
  
  for(; *text; text++)
    {
      if (*text == '\\') fprintf(conn->output_stream, "\\\\");
      else if (*text == '(') fprintf(conn->output_stream, "\\(");
      else if (*text == ')') fprintf(conn->output_stream, "\\)");
      else fprintf(conn->output_stream, "%c", *text);
    }
  if (direction == TGPS_TEXT_DIRECTION_RIGHT_JUSTIFIED ||
      direction == TGPS_TEXT_DIRECTION_UP_JUSTIFIED ||
      direction == TGPS_TEXT_DIRECTION_LEFT_JUSTIFIED ||
      direction == TGPS_TEXT_DIRECTION_DOWN_JUSTIFIED) 
    fprintf(conn->output_stream, ") tj re\n");
  else
    fprintf(conn->output_stream, ") t re\n");
}

/* Print axis */
void tgps_print_axis(Tgps conn)
{
  if (grid != 0)
    {
      double lat_step, lon_step, lat, lon;
      char buffer[20];

      lat_step = (latitude_max - latitude_min) / grid;
      lon_step = (longitude_max - longitude_min) / grid;
      for(lat = latitude_min;
	  lat < latitude_max - lat_step / 2;
	  lat += lat_step)
	{
	  for(lon = longitude_min;
	      lon < longitude_max - lon_step / 2;
	      lon += lon_step)
	    {
	      tgps_ps_draw_line(conn, lat, lon, lat, lon + lon_step);
	      tgps_ps_draw_line(conn, lat, lon, lat + lat_step, lon);
	    }
	  tgps_ps_draw_line(conn, lat, longitude_max,
			    lat + lat_step, longitude_max);
	}
      for(lon = longitude_min;
	  lon < longitude_max - lon_step / 2;
	  lon += lon_step)
	{
	  tgps_ps_draw_line(conn, latitude_max, lon, latitude_max,
			    lon + lon_step);
	}
      for(lat = latitude_min; lat < latitude_max; lat += lat_step)
	{
	  snprintf(buffer, sizeof(buffer), "%g", lat);
	  tgps_ps_draw_text(conn, lat, longitude_min, -1, -2,
			    5, TGPS_TEXT_DIRECTION_RIGHT_JUSTIFIED, buffer);
	  tgps_ps_draw_text(conn, lat, longitude_max, 1, -2,
			    5, TGPS_TEXT_DIRECTION_RIGHT, buffer);
	}
      for(lon = longitude_min; lon < longitude_max; lon += lon_step)
	{
	  snprintf(buffer, sizeof(buffer), "%g", lon);
	  tgps_ps_draw_text(conn, latitude_min, lon, -2, -1,
			    5, TGPS_TEXT_DIRECTION_DOWN, buffer);
	  tgps_ps_draw_text(conn, latitude_max, lon, 2, 1,
			    5, TGPS_TEXT_DIRECTION_UP, buffer);
	}
    }
}

/* Set clippath */
void tgps_set_clip(Tgps conn)
{
  double lat_step, lon_step, lat, lon;

  lat_step = (latitude_max - latitude_min) / grid;
  lon_step = (longitude_max - longitude_min) / grid;

  tgps_ps_draw_line_start(conn, 0, latitude_min, longitude_min);
  for(lat = latitude_min + lat_step; lat < latitude_max; lat += lat_step)
    tgps_ps_draw_line_point(conn, 0, lat, longitude_min);

  for(lon = longitude_min; lon < longitude_max; lon += lon_step)
    tgps_ps_draw_line_point(conn, 0, latitude_max, lon);

  for(lat = latitude_max; lat > latitude_min; lat -= lat_step)
    tgps_ps_draw_line_point(conn, 0, lat, longitude_max);

  for(lon = longitude_max; lon > longitude_min; lon -= lon_step)
    tgps_ps_draw_line_point(conn, 0, latitude_min, lon);

  fprintf(conn->output_stream, "clip\n");
  draw_line = 0;
}

/* Is the point in the visible are of the print area */
int tgps_is_visible(double lat, double lon)
{
  if (lat < latitude_min || lat > latitude_max ||
      lon < longitude_min || lon > longitude_max)
    return 0;
  return 1;
}

/* Write data to output file */
void tgps_print(Tgps conn, const char *format, ...)
{
  va_list ap;

  va_start(ap, format);
  vfprintf(conn->output_stream, format, ap);
  va_end(ap);
}

/* Get line from input file, and put it to input buffer. Returns true if
   successfull, and false if end of file. */
int tgps_gets(Tgps conn)
{
  return (fgets(conn->input_buffer, conn->input_buffer_alloc,
		conn->input_stream) == NULL);
}

int main(int argc, char **argv)
{
  int c, errflg = 0, verbose = 0, version = 0, lines = 0;
  TgpsWaypointStruct w;
  TgpsRouteHeaderStruct r;
  TgpsTrackStruct t;
  TgpsRouteLnkStruct rl;
  TgpsTrackHeaderStruct th;
  TgpsPvtStruct p;
  double next_lat, next_lon;
  unsigned long next_time;
  int prev_outside = 1;
  char *output = "-";
  char *input = "-";
  Tgps conn;

  while ((c = getopt(argc, argv, "o:i:vVhx:y:l:L:g:s:c:btf:")) != EOF)
    {
      switch (c)
	{
	case 'o':
	  output = optarg;
	  break;
	case 'i':
	  input = optarg;
	  break;
	case 'v':
	  verbose++;
	  break;
	case 'V':
	  version = 1;
	  break;
	case 'x':
	  xsize = atoi(optarg);
	  break;
	case 'y':
	  ysize = atoi(optarg);
	  break;
	case 'l':
	  latitude = atof(optarg);
	  break;
	case 'L':
	  longitude = atof(optarg);
	  break;
	case 'g':
	  grid = atoi(optarg);
	  break;
	case 's':
	  scale = atof(optarg);
	  break;
	case 'c':
	  {
	    int i, len;
	    char *ptr;

	    ptr = optarg;
	    for(i = 0; i < NUMBER_OF_SPEEDS; i++)
	      {
		if (sscanf(ptr, " %lf=%li %n", &speed_limits[i],
			   &speed_colors[i],
			   &len) != 2)
		  break;
		fprintf(stderr, "Parsed limit %g = %06lx\n", speed_limits[i],
		       speed_colors[i]);
		ptr += len;
		if (!*ptr)
		  break;
	      }
	    i++;
	    for(; i < NUMBER_OF_SPEEDS; i++)
	      {
		speed_limits[i] = -1.0;
		speed_colors[i] = 0;
	      }
	  }
	  break;
	case 'b':
	  bw_flag = 1;
	  break;
	case 't':
	  t_flag = 1;
	  break;
	case 'f':
	  fontsize = strtod(optarg, NULL);
	  break;
	case 'h':
	case '?':
	  errflg = 1;
	  break;
	}
    }

  if (version)
    {
      if (verbose == 0)
	fprintf(stderr, "T2ps version %s\n", VERSION);
      else if (verbose == 1)
	fprintf(stderr, "T2ps version %s by Tero Kivinen\n", VERSION);
      else if (verbose == 2)
	fprintf(stderr, "T2ps version %s by Tero Kivinen <kivinen@iki.fi>\n",
		VERSION);
      else if (verbose == 3)
	fprintf(stderr, "T2ps version %s by "
		"Tero T. Kivinen <kivinen@iki.fi>\n", VERSION);
      else
	{
	  fprintf(stderr, "T2ps version %s by Tero T. Kivinen "
		  "<kivinen@iki.fi>, http://www.iki.fi/kivinen/\n", VERSION);
	  if (verbose > 10)
	    fprintf(stderr, "No, I will not give you more "
		    "information about myself.\n");
	}
      exit(0);
    }
  if (errflg || argc - optind != 0)
    {
      fprintf(stderr, "Usage: t2ps [-V] [-v] [-x xsize] [-y ysize] "
	      "[-l latitude] [-L longitude] [-g gridlines] [-s scale] "
	      "[-o output_file] [-i input_file] [-f fontsize] "
	      "[-c limit1=color1 limit2=color2 ...] [-b]\n");
      exit(1);
    }
  conn = tgps_open(output, input);

  if (conn == NULL)
    {
      fprintf(stderr, "tgps_open failed\n");
      exit(1);
    }

  conn->verbose = verbose;

  if (latitude > 90.0) latitude = 90.0;
  if (latitude < -90.0) latitude = -90.0;
  if (longitude > 180.0) longitude = 180.0;
  if (longitude < -180.0) longitude = -180.0;

  latitude_max = latitude + ((ysize / 2.0) * scale) /
    degree_in_meters_latitude(latitude);
  latitude_min = latitude - ((ysize / 2.0) * scale) /
    degree_in_meters_latitude(latitude);

  if (latitude_max > 0.0 && latitude_min < 0.0)
    {
      longitude_max =
	longitude + ((xsize / 2.0) * scale) / degree_in_meters_longitude(0.0);
      longitude_min =
	longitude - ((xsize / 2.0) * scale) / degree_in_meters_longitude(0.0);
    }
  else if (latitude_max > 0)
    {
      longitude_max =
	longitude + ((xsize / 2.0) * scale) /
	degree_in_meters_longitude(latitude_min);
      longitude_min =
	longitude - ((xsize / 2.0) * scale) /
	degree_in_meters_longitude(latitude_min);
    }
  else
    {
      longitude_max =
	longitude + ((xsize / 2.0) * scale) /
	degree_in_meters_longitude(latitude_max);
      longitude_min =
	longitude - ((xsize / 2.0) * scale) /
	degree_in_meters_longitude(latitude_max);
    }

  if (latitude_min > 90.0) latitude_min = 90.0;
  if (latitude_min < -90.0) latitude_min = -90.0;
  if (longitude_min > 180.0) longitude_min = 180.0;
  if (longitude_min < -180.0) longitude_min = -180.0;

  if (latitude_max > 90.0) latitude_max = 90.0;
  if (latitude_max < -90.0) latitude_max = -90.0;
  if (longitude_max > 180.0) longitude_max = 180.0;
  if (longitude_max < -180.0) longitude_max = -180.0;

  if (conn->verbose > 1)
    {
      fprintf(stderr, "Bounding box = (%g, %g) <-> (%g, %g)\n",
	      latitude_min, longitude_min,
	      latitude_max, longitude_max);
    }

  tgps_ps_print_header(conn);

  tgps_print_axis(conn);
  tgps_set_clip(conn);

  do {
    conn->input_buffer_len = -1;
    tgps_clear_waypoint(&w);
    tgps_clear_track(&t);
    tgps_clear_route_header(&r);
    tgps_clear_route_link(&rl);
    tgps_clear_track_header(&th);
    if (tgps_get_track(conn, &t))
      {
	tgps_defaults_track(&t);
	next_lat = tgps_semicircle_to_deg(t.lat);
	next_lon = tgps_semicircle_to_deg(t.lon);
	next_time = t.time + TGPS_BASE_TIME;
	if (tgps_is_visible(next_lat, next_lon) || !prev_outside || filling)
	  {
	    tgps_point(conn, t.new_track,
		       next_time, next_lat, next_lon,
		       t_limit);
	    if (tgps_is_visible(next_lat, next_lon))
	      prev_outside = 0;
	    else
	      prev_outside = 1;
	  }
	else
	  {
	    tgps_ps_draw_line_end(conn);
	    tgps_store_point(conn, next_time, next_lat, next_lon);
	  }
      }
    else if (tgps_get_pvt(conn, &p))
      {
	next_lat = tgps_radians_to_deg(p.lat);
	next_lon = tgps_radians_to_deg(p.lon);
	next_time = p.wn_days * 86400 + p.tow - p.leap_scnds;
	if (tgps_is_visible(next_lat, next_lon) || !prev_outside || filling)
	  {
	    tgps_point(conn, t.new_track,
		       next_time, next_lat, next_lon, 60);
	    if (tgps_is_visible(next_lat, next_lon))
	      prev_outside = 0;
	    else
	      prev_outside = 1;
	  }
	else
	  {
	    tgps_ps_draw_line_end(conn);
	    tgps_store_point(conn, next_time, next_lat, next_lon);
	  }
      }
    else if (tgps_get_waypoint(conn, &w))
      {
	double wlat, wlon;

	tgps_ps_draw_line_end(conn);
	wlat = tgps_semicircle_to_deg(w.lat);
	wlon = tgps_semicircle_to_deg(w.lon);
	
	if (tgps_is_visible(wlat, wlon))
	  {
	    tgps_defaults_waypoint(&w);
	    tgps_ps_draw_circle(conn, wlat, wlon, waypointcircle);
	    if (strchr(w.comment_var, ':') != NULL)
	      tgps_ps_draw_text(conn, wlat, wlon,
				textoffsetx, textoffsety,
				fontsize, TGPS_TEXT_DIRECTION_RIGHT,
				w.wpt_ident);
	    else
	      tgps_ps_draw_text(conn, wlat, wlon,
				textoffsetx, textoffsety,
				fontsize, TGPS_TEXT_DIRECTION_RIGHT,
				w.comment_var);
	  }
      }
    else if (tgps_get_route_header(conn, &r))
      {
	/* XXX */
	tgps_defaults_route_header(&r);
      }
    else if (tgps_get_route_link(conn, &rl))
      {
	/* XXX */
	tgps_defaults_route_link(&rl);
      }
    else if (tgps_get_track_hdr(conn, &th))
      {
	tgps_defaults_track_header(&th);
	if (garmin_colors[(int) th.color] != next_color)
	  next_color = garmin_colors[(int) th.color];
	if (th.line_width != next_width)
	  next_width = th.line_width;
	filling = th.fill;
      }
    tgps_free_track_header(&th);
    tgps_free_route_link(&rl);
    tgps_free_waypoint(&w);
    tgps_free_route_header(&r);
    if ((++lines % 1000) == 0 && conn->verbose && isatty(fileno(stderr)))
      fprintf(stderr, "\r%d", lines);
  }  while (conn->input_buffer_len != 0);

  if (conn->verbose && isatty(fileno(stderr)))
      fprintf(stderr, "\r%d\n", lines);

  tgps_ps_print_footer(conn);

  if (tgps_close(conn) < 0)
    {
      fprintf(stderr, "tgps_close failed\n");
      exit(1);
    }
  return 0;
}
