/*
 * routes.c -- GPS protocol route handling.
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/routes.c,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 21:33 Apr 24 2000 kivinen
 *	  Last Modification : 12:21 Aug 19 2003 kivinen
 *	  Last check in     : $Date: 2003/08/18 20:07:59 $
 *	  Revision number   : $Revision: 1.7 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.85
 *	  Edit time	    : 26 min
 *
 *	  Description       : GPS protocol route handling
 *
 *	  $Log: routes.c,v $
 *	  Revision 1.7  2003/08/18 20:07:59  kivinen
 *	  	Added support for GPS V. Fixed packet size and actual size
 *	  	comparisions.
 *
 *	  Revision 1.6  2000/08/05 00:43:46  kivinen
 *	  	Updated to new interface.
 *
 *	  Revision 1.5  2000/07/21 22:16:25  kivinen
 *	  	Moved stuff to data.c.
 *
 *	  Revision 1.4  2000/07/12 22:20:01  kivinen
 *	  	Final upload support.
 *
 *	  Revision 1.3  2000/07/06 23:07:13  kivinen
 *	  	Added initial upload support.
 *
 *	  Revision 1.2  2000/04/30 01:03:51  kivinen
 *	  	Updated to rev 03 document.
 *
 *	  Revision 1.1  2000/04/29 16:40:27  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#include "tgps.h"
#include "packet.h"
#include "data.h"

int tgps_rte_hdr_data_in_d200(Tgps conn, TgpsPacket packet, TgpsRouteHeader r);
int tgps_rte_hdr_data_in_d201(Tgps conn, TgpsPacket packet, TgpsRouteHeader r);
int tgps_rte_hdr_data_in_d202(Tgps conn, TgpsPacket packet, TgpsRouteHeader r);
TgpsPacket tgps_rte_hdr_data_out_d200(Tgps conn, TgpsRouteHeader r);
TgpsPacket tgps_rte_hdr_data_out_d201(Tgps conn, TgpsRouteHeader r);
TgpsPacket tgps_rte_hdr_data_out_d202(Tgps conn, TgpsRouteHeader r);
int tgps_rte_lnk_data_in_d210(Tgps conn, TgpsPacket packet, TgpsRouteLnk r);
TgpsPacket tgps_rte_lnk_data_out_d210(Tgps conn, TgpsRouteLnk r);

/* Process version 200 route header packet */
int tgps_rte_hdr_data_in_d200(Tgps conn, TgpsPacket packet, TgpsRouteHeader r)
{
  size_t size;
 
  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR, &r->number, 
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 201 route header packet */
int tgps_rte_hdr_data_in_d201(Tgps conn, TgpsPacket packet, TgpsRouteHeader r)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_CHAR, &r->number,
		     TGPS_FORMAT_CHAR_ARRAY, r->comment, 20,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process version 202 route header packet */
int tgps_rte_hdr_data_in_d202(Tgps conn, TgpsPacket packet, TgpsRouteHeader r)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_STRING, &r->rte_ident,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process route header packet */
int tgps_rte_hdr_data_in(Tgps conn, TgpsPacket packet, TgpsRouteHeader r)
{
  int ret;

  tgps_clear_route_header(r);

 if (tgps_is_supported(conn, 'D', 200))
   ret = tgps_rte_hdr_data_in_d200(conn, packet, r);
  else if (tgps_is_supported(conn, 'D', 201))
    ret = tgps_rte_hdr_data_in_d201(conn, packet, r);
  else if (tgps_is_supported(conn, 'D', 202))
    ret = tgps_rte_hdr_data_in_d202(conn, packet, r);
  else if (packet->data_len == 3)
    ret = tgps_rte_hdr_data_in_d200(conn, packet, r);
  else if (packet->data_len == 23)
    ret = tgps_rte_hdr_data_in_d201(conn, packet, r);
  else
    {
      fprintf(stderr, "Unknown route header type\n");
      return 0;
    }
  if (ret != 0)
    conn->packet_count--;
  return ret;
}

/* Send version 200 route header data packet */
TgpsPacket tgps_rte_hdr_data_out_d200(Tgps conn, TgpsRouteHeader r)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_RTE_HDR(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR, r->number, 
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 201 route header data packet */
TgpsPacket tgps_rte_hdr_data_out_d201(Tgps conn, TgpsRouteHeader r)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_RTE_HDR(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_CHAR, r->number,
		     TGPS_FORMAT_CHAR_ARRAY, r->comment, 20,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send version 202 route header data packet */
TgpsPacket tgps_rte_hdr_data_out_d202(Tgps conn, TgpsRouteHeader r)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_RTE_HDR(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_STRING, r->rte_ident,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send route header packet */
TgpsPacket tgps_rte_hdr_data_out(Tgps conn, TgpsRouteHeader r)
{
  if (tgps_is_supported(conn, 'D', 200))
    return tgps_rte_hdr_data_out_d200(conn, r);
  else if (tgps_is_supported(conn, 'D', 201))
    return tgps_rte_hdr_data_out_d201(conn, r);
  else if (tgps_is_supported(conn, 'D', 202))
    return tgps_rte_hdr_data_out_d202(conn, r);
  else
    {
      fprintf(stderr, "Unknown route header type\n");
      return NULL;
    }
}

/* Process version 210 route link type */
int tgps_rte_lnk_data_in_d210(Tgps conn, TgpsPacket packet, TgpsRouteLnk r)
{
  size_t size;

  size = tgps_decode(packet->data, packet->data_len,
		     TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
		     TGPS_FORMAT_WORD, &r->class,
		     TGPS_FORMAT_BYTE_ARRAY, &r->subclass, 18,
		     TGPS_FORMAT_STRING, &r->ident,
		     TGPS_FORMAT_END);
  if (size == packet->data_len)
    return 1;
  if (size < packet->data_len)
    {
      fprintf(stderr, "Warning, extra junk after packet\n");
      return 1;
    }
  return 0;
}

/* Process route link packet */
int tgps_rte_lnk_data_in(Tgps conn, TgpsPacket packet, TgpsRouteLnk r)
{
  int ret;

  tgps_clear_route_link(r);

  if (tgps_is_supported(conn, 'D', 210))
    ret = tgps_rte_lnk_data_in_d210(conn, packet, r);
  else
    {
      fprintf(stderr, "Unknown route link type\n");
      return 0;
    }
  if (ret != 0)
    conn->packet_count--;
  return ret;
}

/* Send version 210 route link data packet */
TgpsPacket tgps_rte_lnk_data_out_d210(Tgps conn, TgpsRouteLnk r)
{
  TgpsPacket p = NULL;
  size_t size;

  p = tgps_get_packet(conn, 255);
  size = tgps_encode(p->data, 255,
		     TGPS_FORMAT_BYTE, TGPS_PID_RTE_HDR(conn),
		     TGPS_FORMAT_LEN,
		     TGPS_FORMAT_WORD, r->class,
		     TGPS_FORMAT_BYTE_ARRAY, r->subclass, 18,
		     TGPS_FORMAT_STRING, r->ident,
		     TGPS_FORMAT_END);
  if (size <= 0)
    {
      fprintf(stderr, "Internal error, packet buffer too "
	      "short for packet\n");
      exit(1);
    }
  else
    {
      p->data_len = size;
    }
  return p;
}

/* Send route link packet */
TgpsPacket tgps_rte_lnk_data_out(Tgps conn, TgpsRouteLnk r)
{
  if (tgps_is_supported(conn, 'D', 210))
    return tgps_rte_lnk_data_out_d210(conn, r);
  else
    {
      fprintf(stderr, "Unknown route link type\n");
      return NULL;
    }
}
