/*
 * position.c -- GPS protocol position handling.
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/position.c,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 21:36 Apr 24 2000 kivinen
 *	  Last Modification : 14:48 Aug  9 2003 kivinen
 *	  Last check in     : $Date: 2003/08/18 20:07:47 $
 *	  Revision number   : $Revision: 1.5 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.58
 *	  Edit time	    : 13 min
 *
 *	  Description       : GPS protocol position handling
 *
 *	  $Log: position.c,v $
 *	  Revision 1.5  2003/08/18 20:07:47  kivinen
 *	  	Added support for GPS V. Fixed packet size and actual size
 *	  	comparisions.
 *
 *	  Revision 1.4  2000/08/05 00:44:03  kivinen
 *	  	Updated to new interface.
 *
 *	  Revision 1.3  2000/07/26 17:08:50  kivinen
 *	  	Added pvt and position upload support.
 *
 *	  Revision 1.2  2000/07/06 23:07:05  kivinen
 *	  	Added initial upload support.
 *
 *	  Revision 1.1  2000/04/29 16:40:24  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#include "tgps.h"
#include "data.h"
#include "packet.h"

/* Process position packet */
int tgps_position_data_in(Tgps conn, TgpsPacket packet, TgpsPosition p)
{
  if (tgps_is_supported(conn, 'D', 700) || packet->data_len == 18)
    {
      size_t size;

      size = tgps_decode(packet->data, packet->data_len,
			 TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
			 TGPS_FORMAT_RADIAN, &p->lat, &p->lon,
			 TGPS_FORMAT_END);
      if (size == packet->data_len)
	{
	  conn->packet_count--;
	  if (conn->packet_count == 0)
	    conn->xfer_done = 1;
	  return 1;
	}
      if (size < packet->data_len)
	{
	  conn->packet_count--;
	  if (conn->packet_count == 0)
	    conn->xfer_done = 1;
	  fprintf(stderr, "Warning, extra junk after packet\n");
	  return 1;
	}
      return 0;
    }
  else
    {
      fprintf(stderr, "Unknown position data\n");
      return 0;
    }
}

/* Send position packet */
TgpsPacket tgps_position_data_out(Tgps conn, TgpsPosition pos)
{
  TgpsPacket p = NULL;
  size_t size;

  if (tgps_is_supported(conn, 'D', 700))
    {
      p = tgps_get_packet(conn, 255);
      size = tgps_encode(p->data, 255,
			 TGPS_FORMAT_BYTE, TGPS_PID_POSITION_DATA(conn),
			 TGPS_FORMAT_LEN,
			 TGPS_FORMAT_RADIAN, pos->lat, pos->lon,
			 TGPS_FORMAT_END);
      if (size <= 0)
	{
	  fprintf(stderr, "Internal error, packet buffer too "
		  "short for packet\n");
	  exit(1);
	}
      else
	{
	  p->data_len = size;
	}
    }
  else
    {
      fprintf(stderr, "Unknown position protocol format\n");
    }
  return p;
}

/* Process PVT packet */
int tgps_pvt_data_in(Tgps conn, TgpsPacket packet, TgpsPvt p)
{
  if (tgps_is_supported(conn, 'D', 800) || packet->data_len == 66)
    {
      size_t size;

      size = tgps_decode(packet->data, packet->data_len,
			 TGPS_FORMAT_BYTE, NULL, TGPS_FORMAT_BYTE, NULL,
			 TGPS_FORMAT_FLOAT, &p->alt,
			 TGPS_FORMAT_FLOAT, &p->epe,
			 TGPS_FORMAT_FLOAT, &p->eph,
			 TGPS_FORMAT_FLOAT, &p->epv,
			 TGPS_FORMAT_INT, &p->fix,
			 TGPS_FORMAT_DOUBLE, &p->tow,
			 TGPS_FORMAT_RADIAN, &p->lat, &p->lon,
			 TGPS_FORMAT_FLOAT, &p->east,
			 TGPS_FORMAT_FLOAT, &p->north,
			 TGPS_FORMAT_FLOAT, &p->up,
			 TGPS_FORMAT_FLOAT, &p->msl_hght,
			 TGPS_FORMAT_INT, &p->leap_scnds,
			 TGPS_FORMAT_LONG, &p->wn_days,
			 TGPS_FORMAT_END);
      if (size == packet->data_len)
	return 1;
      if (size < packet->data_len)
	{
	  fprintf(stderr, "Warning, extra junk after packet\n");
	  return 1;
	}
      return 0;
    }
  else
    {
      fprintf(stderr, "Unknown pvt data\n");
      return 0;
    }
}

/* Send pvt packet */
TgpsPacket tgps_pvt_data_out(Tgps conn, TgpsPvt pvt)
{
  TgpsPacket p = NULL;
  size_t size;

  if (tgps_is_supported(conn, 'D', 700))
    {
      p = tgps_get_packet(conn, 255);
      size = tgps_encode(p->data, 255,
			 TGPS_FORMAT_BYTE, TGPS_PID_PVT_DATA(conn),
			 TGPS_FORMAT_LEN,
			 TGPS_FORMAT_FLOAT, pvt->alt,
			 TGPS_FORMAT_FLOAT, pvt->epe,
			 TGPS_FORMAT_FLOAT, pvt->eph,
			 TGPS_FORMAT_FLOAT, pvt->epv,
			 TGPS_FORMAT_INT, pvt->fix,
			 TGPS_FORMAT_DOUBLE, pvt->tow,
			 TGPS_FORMAT_RADIAN, pvt->lat, pvt->lon,
			 TGPS_FORMAT_FLOAT, pvt->east,
			 TGPS_FORMAT_FLOAT, pvt->north,
			 TGPS_FORMAT_FLOAT, pvt->up,
			 TGPS_FORMAT_FLOAT, pvt->msl_hght,
			 TGPS_FORMAT_INT, pvt->leap_scnds,
			 TGPS_FORMAT_LONG, pvt->wn_days,
			 TGPS_FORMAT_END);
      if (size <= 0)
	{
	  fprintf(stderr, "Internal error, packet buffer too "
		  "short for packet\n");
	  exit(1);
	}
      else
	{
	  p->data_len = size;
	}
    }
  else
    {
      fprintf(stderr, "Unknown pvt protocol format\n");
    }
  return p;
}
