/*
 * ieee-double.c -- IEEE double routines
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/ieee-double.c,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 17:01 Apr 20 2000 kivinen
 *	  Last Modification : 19:47 Aug  4 2000 kivinen
 *	  Last check in     : $Date: 2000/08/05 00:45:19 $
 *	  Revision number   : $Revision: 1.2 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.147
 *	  Edit time	    : 97 min
 *
 *	  Description       : IEEE double routines
 *
 *	  $Log: ieee-double.c,v $
 *	  Revision 1.2  2000/08/05 00:45:19  kivinen
 *	  	Removed warnings.
 *
 *	  Revision 1.1  2000/04/29 16:40:13  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#include "tgps.h"
#include "ieee-double.h"
#include <math.h>

/* Swap n byte buffer to reverse order. */
void tgps_swap_buffer(unsigned char *from_buf, unsigned char *to_buf,
		      size_t l);

/* Convert IEEE double format number from buffer to machine representation */
double tgps_buf2d(unsigned char *buf)
{
  int exponent;
  double mantissa;
  int i;
  
  exponent = ((buf[0] & 0x7f) << 4) | (buf[1] >> 4);
  exponent -= 1023;
  mantissa = 0.0;
  for(i = 7; i > 1; i--)
    {
      mantissa += (double) buf[i];
      mantissa /= 256.0;
    }
  mantissa += (double) (buf[1] & 0x0f);
  mantissa /= 16.0;
  if (exponent != -1023)
    mantissa += 1.0;
  else
    mantissa *= 2;
  if (buf[0] >> 7)
    mantissa = copysign(mantissa, -1.0);
  if (exponent == 1024 && mantissa != 1.0)
    {
      mantissa = scalb(mantissa, exponent);
      return mantissa - mantissa;
    }
  return scalb(mantissa, exponent);
}

/* Swap n byte buffer to reverse order. */
void tgps_swap_buffer(unsigned char *from_buf, unsigned char *to_buf, size_t l)
{
  int i;

  for(i = 0; i < l; i++)
    to_buf[i] = from_buf[l - 1 - i];
}

/* Convert IEEE double format number from buffer to machine representation, LSB
   first. */
double tgps_buf2d_lsb(unsigned char *buf)
{
  unsigned char buf2[8];

  tgps_swap_buffer(buf, buf2, 8);
  return tgps_buf2d(buf2);
}

/* Convert machine double to IEEE double and store it to buffer. Buffer must be
   at least 8 characters long. */
void tgps_d2buf(unsigned char *buf, double d)
{
  int exponent, i, sign;
  double mantissa;

  if (isinf(d))
    {
      exponent = 1024;
      mantissa = 0.0;
    }
  else if (isnan(d))
    {
      exponent = 1024;
      mantissa = 0.1;
    }
  else
    {
      exponent = ilogb(d);
      mantissa = d / scalb(1.0, exponent);
    }
  if (copysign(1.0, d) < 0.0)
    {
      sign = 1;
      mantissa = -mantissa;
    }
  else
    {
      sign = 0;
    }
  exponent += 1023;
  if (exponent <= 0)
    {
      if (exponent < -1000)
	{
	  mantissa = 1.0;
	  exponent = 0;
	}
      else
	{
	  while (exponent <= 0)
	    {
	      mantissa /= 2;
	      exponent++;
	    }
	  exponent--;
	  mantissa += 1.0;
	}
    }
  buf[0] = (sign << 7) | (exponent >> 4);
  mantissa -= 1.0;
  mantissa *= 16.0;
  buf[1] = (exponent & 0x0f) << 4 | ((int) mantissa);
  mantissa -= (double) (int) mantissa;
  for(i = 2; i < 8; i++)
    {
      mantissa *= 256.0;
      buf[i] = (int) mantissa;
      mantissa -= (double) (int) mantissa;
    }
}

/* Convert machine double to IEEE double and store it to buffer. Buffer must be
   at least 8 characters long. LSB first. */
void tgps_d2buf_lsb(unsigned char *buf, double d)
{
  unsigned char buf2[8];

  tgps_d2buf(buf2, d);
  tgps_swap_buffer(buf2, buf, 8);
}

/* Convert IEEE float format number from buffer to machine representation */
float tgps_buf2f(unsigned char *buf)
{
  int exponent;
  float mantissa;
  int i;
  
  exponent = ((buf[0] & 0x7f) << 1) | (buf[1] >> 7);
  exponent -= 127;
  mantissa = 0.0;
  for(i = 3; i > 1; i--)
    {
      mantissa += (float) buf[i];
      mantissa /= 256.0;
    }
  mantissa += (float) (buf[1] & 0x7f);
  mantissa /= 128.0;
  if (exponent != -127)
    mantissa += 1.0;
  else
    mantissa *= 2;
  if (buf[0] >> 7)
    mantissa = copysignf(mantissa, -1.0);
  if (exponent == 128 && mantissa != 1.0)
    {
      mantissa = scalbf(mantissa, exponent);
      return mantissa - mantissa;
    }
  return scalbf(mantissa, exponent);
}

/* Convert IEEE float format number from buffer to machine representation, LSB
   first.  */
float tgps_buf2f_lsb(unsigned char *buf)
{
  unsigned char buf2[4];

  tgps_swap_buffer(buf, buf2, 4);
  return tgps_buf2f(buf2);
}

/* Convert machine float to IEEE double and store it to buffer. Buffer must be
   at least 4 characters long. */
void tgps_f2buf(unsigned char *buf, float f)
{
  int exponent, i, sign;
  float mantissa;

  if (isinff(f))
    {
      exponent = 128;
      mantissa = 0.0;
    }
  else if (isnanf(f))
    {
      exponent = 128;
      mantissa = 0.1;
    }
  else
    {
      exponent = ilogbf(f);
      mantissa = f / scalbf(1.0, exponent);
    }
  if (copysignf(1.0, f) < 0.0)
    {
      sign = 1;
      mantissa = -mantissa;
    }
  else
    {
      sign = 0;
    }
  exponent += 127;
  if (exponent <= 0)
    {
      if (exponent < -1000)
	{
	  mantissa = 1.0;
	  exponent = 0;
	}
      else
	{
	  while (exponent <= 0)
	    {
	      mantissa /= 2;
	      exponent++;
	    }
	  exponent--;
	  mantissa += 1.0;
	}
    }
  buf[0] = (sign << 7) | (exponent >> 1);
  mantissa -= 1.0;
  mantissa *= 128.0;
  buf[1] = (exponent & 0x1) << 7 | ((int) mantissa);
  mantissa -= (float) (int) mantissa;
  for(i = 2; i < 4; i++)
    {
      mantissa *= 256.0;
      buf[i] = (int) mantissa;
      mantissa -= (float) (int) mantissa;
    }
}

/* Convert machine float to IEEE double and store it to buffer. Buffer must be
   at least 4 characters long. LSB first. */
void tgps_f2buf_lsb(unsigned char *buf, float f)
{
  unsigned char buf2[4];

  tgps_f2buf(buf2, f);
  tgps_swap_buffer(buf2, buf, 4);
}

#ifdef IEEE_TESTING
void print_d(char *txt, double d)
{
  int i;
  unsigned char *p;

  printf("%s0x", txt);
#ifdef WORDS_BIGENDIAN
  p = (unsigned char *) &d;
  for(i = 0; i < 8; i++)
    printf("%02x", *p++);
#else  /* WORDS_BIGENDIAN */
  p = ((unsigned char *) &d) + 7;
  for(i = 0; i < 8; i++)
    printf("%02x", *p--);
#endif /* WORDS_BIGENDIAN */
  printf(" = %g\n", d);
}

void print_f(char *txt, float f)
{
  int i;
  unsigned char *p;

  printf("%s0x", txt);
#ifdef WORDS_BIGENDIAN
  p = (unsigned char *) &f;
  for(i = 0; i < 4; i++)
    printf("%02x", *p++);
#else  /* WORDS_BIGENDIAN */
  p = ((unsigned char *) &f) + 3;
  for(i = 0; i < 4; i++)
    printf("%02x", *p--);
#endif /* WORDS_BIGENDIAN */
  printf(" = %f\n", f);
}

void print_buf(char *txt, unsigned char *buf, size_t len)
{
  int i;

  printf("%s0x", txt);
  for(i = 0; i < len; i++)
    printf("%02x", buf[i]);
  printf("\n");
}

int main(int argc, char **argv)
{
  double d, n;
  float f, m;
  int i, j;
  unsigned char dbuf[8], fbuf[4], *p;

  for(i = 0; i < 100000; i++)
    {
      if (argc == 1)
	{
	  for(j = 0, p = (unsigned char *) &d; j < 8; j++, p++)
	    *p = random();
	  memcpy(&f, &d, 4);
	}
      else
	{
	  d = strtod(argv[1], NULL);
	  f = strtod(argv[1], NULL);
	  i = -1;
	}

      tgps_d2buf(dbuf, d);
      n = tgps_buf2d(dbuf);

      tgps_f2buf(fbuf, f);
      m = tgps_buf2f(fbuf);
      if ((memcmp(&d, &n, 8) != 0 || d != n))
	{
	  if (!isnan(d) || !isnan(n))
	    {
	      printf("Error, d and n not equal\n");
	      print_d("d = ", d);
	      print_d("n = ", n);
	      print_buf("b = ", dbuf, 8);
	      exit(1);
	    }
	  else
	    {
#if 0
	      printf("Error, d and n not equal, but both are NaN\n");
	      print_d("d = ", d);
	      print_d("n = ", n);
	      print_buf("b = ", dbuf, 8);
#endif
	    }
	}
      if ((memcmp(&f, &m, 4) != 0 || f != m))
	{
	  if (!isnan(f) || !isnan(m))
	    {
	      printf("Error, f and m not equal\n");
	      print_f("f = ", f);
	      print_f("m = ", m);
	      print_buf("b = ", fbuf, 4);
	      exit(1);
	    }
	  else
	    {
#if 0
	      printf("Error, f and m not equal, but both are NaN\n");
	      print_f("f = ", f);
	      print_f("m = ", m);
	      print_buf("b = ", fbuf, 4);
#endif
	    }
	}
      if (i == -1)
	{
	  printf("Ok\n");
	  print_d("d = ", d);
	  print_d("n = ", n);
	  print_buf("b = ", dbuf, 8);
	  print_f("f = ", f);
	  print_f("m = ", m);
	  print_buf("b = ", fbuf, 4);
	  break;
	}
    }
  return 0;
}
#endif /* IEEE_TESTING */

