/*
 * display.c -- GPS protocol display code.
 *
 * Copyright (c) 2000 Tero Kivinen <kivinen@iki.fi>
 */
/*
 *        Program: tgps
 *	  $Source: /u/kivinen/gps/tgps/RCS/display.c,v $
 *	  Author : $Author: kivinen $
 *
 *	  Creation          : 21:41 Apr 24 2000 kivinen
 *	  Last Modification : 00:51 Jan  8 2004 kivinen
 *	  Last check in     : $Date: 2003/08/18 20:10:01 $
 *	  Revision number   : $Revision: 1.13 $
 *	  State             : $State: Exp $
 *	  Version	    : 1.501
 *	  Edit time	    : 166 min
 *
 *	  Description       : GPS protocol display code
 *
 *	  $Log: display.c,v $
 *	  Revision 1.13  2003/08/18 20:10:01  kivinen
 *	  	Added support for GPS V. Fixed packet size and actual size
 *	  	comparisions.
 *
 *	  Revision 1.12  2001/08/23 15:41:35  kivinen
 *	  	Disabled printing of proximity field if it is bigger than
 *	  	1e+25. Fixed bug that printed waypoint subclass as signed
 *	  	values. Added code that will ignore variable length string
 *	  	fields if they are empty. Added printing of extra data in
 *	  	track and almanac records.
 *
 *	  Revision 1.11  2000/10/02 15:43:12  kivinen
 *	  	Changed to use tgps_print.
 *
 *	  Revision 1.10  2000/08/05 00:45:14  kivinen
 *	  	Removed warnings.
 *
 *	  Revision 1.9  2000/07/23 20:23:18  kivinen
 *	  	Added fix string conversion. Changed pvt format to be -xxls
 *	  	instead of +xxls.
 *
 *	  Revision 1.8  2000/07/18 20:23:10  kivinen
 *	  	Removed empty line before new track starts.
 *
 *	  Revision 1.7  2000/07/17 18:09:10  kivinen
 *	  	Added new format for newtrack.
 *
 *	  Revision 1.6  2000/07/10 21:39:45  kivinen
 *	  	Added code to ignore compressed tracks for v1 tracks also.
 *
 *	  Revision 1.5  2000/07/07 20:40:53  kivinen
 *	  	Added support for ignoring compressed tracks.
 *
 *	  Revision 1.4  2000/07/06 23:06:09  kivinen
 *	  	Added initial upload support.
 *
 *	  Revision 1.3  2000/04/30 01:33:40  kivinen
 *	  	Changed invalid float detection code.
 *
 *	  Revision 1.2  2000/04/30 01:03:04  kivinen
 *	  	Updated to rev 03 document.
 *
 *	  Revision 1.1  2000/04/29 16:40:00  kivinen
 *	  	Created.
 *
 *	  $EndLog$
 */

#include "tgps.h"
#include "display.h"

/* Print position */
void tgps_print_pos(Tgps conn, double d);

/* Print waypoint structure */
void tgps_print_waypoint_v1(Tgps conn, TgpsWaypoint w);

/* Print waypoint structure in gpstrans format */
void tgps_print_waypoint_gpstrans(Tgps conn, TgpsWaypoint w);

/* Print route header structure in gpstrans format */
void tgps_print_route_header_gpstrans(Tgps conn, TgpsRouteHeader r);

/* Print route header structure */
void tgps_print_route_header_v1(Tgps conn, TgpsRouteHeader r);

/* Print track point structure in gpstrans format */
void tgps_print_track_gpstrans(Tgps conn, TgpsTrack t);

/* Print track point structure */
void tgps_print_track_v1(Tgps conn, TgpsTrack t);

/* Print almanac structure in gpstrans format */
void tgps_print_almanac_gpstrans(Tgps conn, TgpsAlmanac a);

/* Print almanac structure */
void tgps_print_almanac_v1(Tgps conn, TgpsAlmanac a);

/* Print time structure in gpstrans format */
void tgps_print_time_gpstrans(Tgps conn, TgpsTime t);

/* Print time structure */
void tgps_print_time_v1(Tgps conn, TgpsTime t);

/* Set the system time */
void tgps_set_time(Tgps conn, TgpsTime t);

/* Print position structure */
void tgps_print_position_v1(Tgps conn, TgpsPosition p);

/* Print pvt structure */
void tgps_print_pvt_v1(Tgps conn, TgpsPvt p);

/* Print header */
void tgps_print_header_v1(Tgps conn);

/* Print header in gpstrans format */
void tgps_print_header_gpstrans(Tgps conn);




/* Print position */
void tgps_print_pos(Tgps conn, double d)
{
  double d1;
  int i1, i2;

  if (d < 0.0)
    {
      tgps_print(conn, "-");
      d = -d;
    }
  switch (conn->degree_format)
    {
    case TGPS_DEG_MIN_SEC:
      i1 = d;
      i2 = (d - i1) * 60;
      d1 = (d - (i1 + i2 / 60.0)) * 3600.0;
      if (d1 > 59.95) { d1 = 0.0; i2++; }
      if (i2 >= 60) { i2 = 0; i1++; }
      tgps_print(conn, "%d%02d'%04.1f\"", i1, i2, d1);
      break;
    case TGPS_DEG_MIN:
      i1 = d;
      d1 = (d - i1) * 60;
      if (d1 > 59.9995) { d1 = 0.0; i1++; }
      tgps_print(conn, "%d%06.3f'", i1, d1);
      break;
    case TGPS_DEG:
      tgps_print(conn, "%.7f", d);
      break;
    case TGPS_DEG_NONE:
      fprintf(stderr, "Internal error: Degree format is none\n");
      exit(1);
      break;
    }
}

/* Print waypoint structure */
void tgps_print_waypoint_v1(Tgps conn, TgpsWaypoint w)
{
  const char *comma = "";
  int i;

  tgps_print(conn, "W1\t");

  if (w->wpt_ident != NULL)
    tgps_print(conn, "%s\t", w->wpt_ident);
  else if (!tgps_is_empty(w->ident))
    tgps_print(conn, "%-6.6s\t", w->ident);
  else
    tgps_print(conn, "\t");

  if (w->lnk_ident != NULL)
    tgps_print(conn, "-> %-13s\t", w->lnk_ident);
  else if (w->comment_var)
    tgps_print(conn, "%-16s\t", w->comment_var);
  else
    tgps_print(conn, "%-16s\t", w->comment);

  tgps_print_pos(conn, tgps_semicircle_to_deg(w->lat));
  tgps_print(conn, "\t");
  tgps_print_pos(conn, tgps_semicircle_to_deg(w->lon));
  tgps_print(conn, "\t");

  if (w->smbl != TGPS_INVALID_CHAR)
    {
      tgps_print(conn, "%sSmbl=%s", comma,
		 tgps_symbol_to_string(w->smbl_type, w->smbl));
      comma = ", ";
    }
  else if (w->smbl16 != TGPS_INVALID_INT)
    {
      tgps_print(conn, "%sSmbl=%s", comma, 
	      tgps_symbol_to_string(w->smbl_type, w->smbl16));
      comma = ", ";
    }

  switch (w->display)
    {
    case 0:
    case 3:
      tgps_print(conn, "%sDspl=S+N", comma); comma = ", "; break;
    case 1:
      tgps_print(conn, "%sDspl=S", comma); comma = ", "; break;
    case 2:
    case 5:
      tgps_print(conn, "%sDspl=S+C", comma); comma = ", "; break;
    }

  if (w->color != TGPS_INVALID_CHAR &&
      w->color != 0)
    {
      tgps_print(conn, "%sColor=%s", comma,
		 tgps_color_to_string(w->color));
      comma = ", ";
    }

  if (w->attribute != TGPS_INVALID_CHAR)
    {
      tgps_print(conn, "%sAttr=%d", comma, w->attribute);
      comma = ", ";
    }

  if (w->proximity > TGPS_INVALID_FLOAT_TEST && w->proximity != 0.0 &&
      w->proximity < 0.9e+25)
    {
      tgps_print(conn, "%sPrxDst=%g", comma, w->proximity);
      comma = ", ";
    }

  if (w->class != TGPS_INVALID_CHAR)
    {
      tgps_print(conn, "%sClass=%d", comma, w->class);
      comma = ", ";
    }

  for(i = sizeof(w->subclass) - 1; i >= 0; i--)
    if (w->subclass[i])
      break;
  if (i != -1)
    {
      int j;

      tgps_print(conn, "%sSubclass=0x", comma);
      for(j = 0; j <= i; j++)
	tgps_print(conn, "%02x", (unsigned char) w->subclass[j]);
      comma = ", ";
    }

  if (!tgps_is_empty(w->cc))
    {
      tgps_print(conn, "%sCc=%s", comma, w->cc);
      comma = ", ";
    }

  if (w->facility_var != NULL && strlen(w->facility_var) != 0)
    {
      tgps_print(conn, "%sFacility=%s", comma, w->facility_var);
      comma = ", ";
    }

  if (w->city_var != NULL && strlen(w->city_var) != 0)
    {
      tgps_print(conn, "%sCity=%s", comma, w->city_var);
      comma = ", ";
    }
  else if (!tgps_is_empty(w->city))
    {
      tgps_print(conn, "%sCity=%s", comma, w->city);
      comma = ", ";
    }

  if (!tgps_is_empty(w->state))
    {
      tgps_print(conn, "%sState=%s", comma, w->state);
      comma = ", ";
    }

  if (!tgps_is_empty(w->name))
    {
      tgps_print(conn, "%sName=%s", comma, w->name);
      comma = ", ";
    }

  if (w->addr_var != NULL && strlen(w->addr_var) != 0)
    {
      tgps_print(conn, "%sAddress=%s", comma, w->addr_var);
      comma = ", ";
    }

  if (w->cross_road_var != NULL && strlen(w->cross_road_var) != 0)
    {
      tgps_print(conn, "%sCrossRoad=%s", comma, w->cross_road_var);
      comma = ", ";
    }

  if (w->idx != TGPS_INVALID_CHAR)
    {
      tgps_print(conn, "%sPrxIdx=%d", comma, w->idx);
      comma = ", ";
    }

  if (w->altf > TGPS_INVALID_FLOAT_TEST)
    {
      tgps_print(conn, "%sAlt=%g", comma, (double) w->altf);
      comma = ", ";
    }
  else if (w->alt != TGPS_INVALID_INT)
    {
      tgps_print(conn, "%sAlt=%d", comma, w->alt);
      comma = ", ";
    }

  if (w->dpth > TGPS_INVALID_FLOAT_TEST)
    {
      tgps_print(conn, "%sDepth=%g", comma, (double) w->dpth);
      comma = ", ";
    }

  if (w->ete != 0xffffffff)
    {
      tgps_print(conn, "%sEte=%ul", comma, w->ete);
      comma = ", ";
    }


  if (w->unused_creation_time != 0 &&
      w->unused_creation_time != 0x7fffffff &&
      w->unused_creation_time != 0xffffffff)
    {
      char time_buf[40];
      unsigned long tim;
      struct tm *tmptr;
      time_t t;
      
      t = tim;
      tmptr = gmtime(&t);
      strftime(time_buf, sizeof(time_buf), "%F %T GMT", tmptr);

      tgps_print(conn, "%sTime=%s", comma, time_buf);
      comma = ", ";
    }
  tgps_print(conn, "\n");
}

/* Print waypoint structure in gpstrans format */
void tgps_print_waypoint_gpstrans(Tgps conn, TgpsWaypoint w)
{
  char time_buf[40];
  struct tm *tmptr;
  time_t time;

  time = w->unused_creation_time + TGPS_BASE_TIME;
  tmptr = localtime(&time);
  strftime(time_buf, sizeof(time_buf), "%m/%d/%Y %T", tmptr);

  tgps_print(conn, "W\t");
  if (!tgps_is_empty(w->ident))
    tgps_print(conn, "%-6.6s\t", w->ident);
  else if (w->wpt_ident != NULL)
    tgps_print(conn, "%s\t", w->wpt_ident);
  else
    tgps_print(conn, "\t");
  tgps_print(conn, "%-40s\t", w->comment);

  tgps_print(conn, "%s\t", time_buf);
  
  tgps_print_pos(conn, tgps_semicircle_to_deg(w->lat));
  tgps_print(conn, "\t");
  tgps_print_pos(conn, tgps_semicircle_to_deg(w->lon));
  tgps_print(conn, "\n");
}

/* Print waypoint structure */
void tgps_print_waypoint(Tgps conn, TgpsWaypoint w)
{
  switch (conn->data_version)
    {
    case TGPS_DATA_VERSION_GPSTRANS:
      tgps_print_waypoint_gpstrans(conn, w);
      break;
    case TGPS_DATA_VERSION_TGPS_V1:
      tgps_print_waypoint_v1(conn, w);
      break;
    }
  fflush(conn->output_stream);
}

/* Print route header structure in gpstrans format */
void tgps_print_route_header_gpstrans(Tgps conn, TgpsRouteHeader r)
{
  tgps_print(conn, "R\t");
  if (r->number != TGPS_INVALID_CHAR)
    tgps_print(conn, "%d\t", r->number);
  else
    tgps_print(conn, "0\t");
  if (!tgps_is_empty(r->comment))
    tgps_print(conn, "%s", r->comment);
  else if (r->rte_ident != NULL)
    tgps_print(conn, "%s", r->rte_ident);
  tgps_print(conn, "\n");
}

/* Print route header structure */
void tgps_print_route_header_v1(Tgps conn, TgpsRouteHeader r)
{
  tgps_print(conn, "R1\t");
  if (r->number != TGPS_INVALID_CHAR)
    tgps_print(conn, "%d\t", r->number);
  else
    tgps_print(conn, "0\t");

  if (!tgps_is_empty(r->comment))
    tgps_print(conn, "%-20.20s", r->comment);
  else if (r->rte_ident != NULL)
    tgps_print(conn, "%s", r->rte_ident);
  tgps_print(conn, "\n");
}

/* Print route header structure */
void tgps_print_route_header(Tgps conn, TgpsRouteHeader r)
{
  switch (conn->data_version)
    {
    case TGPS_DATA_VERSION_GPSTRANS:
      tgps_print_route_header_gpstrans(conn, r);
      break;
    case TGPS_DATA_VERSION_TGPS_V1:
      tgps_print_route_header_v1(conn, r);
      break;
    }
  fflush(conn->output_stream);
}

/* Print route link structure */
void tgps_print_route_link(Tgps conn, TgpsRouteLnk r)
{
  const char *comma = "";
  int i;

  tgps_print(conn, "K1\t");
  if (r->ident != NULL)
    tgps_print(conn, "%-20s\t", r->ident);
  else
    tgps_print(conn, "\t");

  if (r->class != TGPS_INVALID_CHAR)
    {
      tgps_print(conn, "%sClass=%d", comma, r->class);
      comma = ", ";
    }

  for(i = sizeof(r->subclass) - 1; i >= 0; i--)
    if (r->subclass[i])
      break;
  if (i != -1)
    {
      int j;

      tgps_print(conn, "%sSubclass=0x", comma);
      for(j = 0; j <= i; j++)
	tgps_print(conn, "%02x", (unsigned char) r->subclass[j]);
      comma = ", ";
    }
  tgps_print(conn, "\n");
  fflush(conn->output_stream);
}

/* Print track point structure in gpstrans format */
void tgps_print_track_gpstrans(Tgps conn, TgpsTrack t)
{
  char time_buf[40];
  struct tm *tmptr;
  time_t time;

  if (conn->flags && TGPS_FLAGS_IGNORE_COMPRESSED_TRACKS &&
      t->time == 0)
    return;
  if (t->new_track)
    tgps_print(conn, "\n");
  time = t->time + TGPS_BASE_TIME;
  tmptr = localtime(&time);
  strftime(time_buf, sizeof(time_buf), "%m/%d/%Y %T", tmptr);

  tgps_print(conn, "T\t%s\t", time_buf);
  tgps_print_pos(conn, tgps_semicircle_to_deg(t->lat));
  tgps_print(conn, "\t");
  tgps_print_pos(conn, tgps_semicircle_to_deg(t->lon));
  tgps_print(conn, "\n");
}

/* Print track point structure */
void tgps_print_track_v1(Tgps conn, TgpsTrack t)
{
  const char *comma = "\t";
  char time_buf[40];
  struct tm *tmptr;
  time_t time;
  int i;

  if (conn->flags && TGPS_FLAGS_IGNORE_COMPRESSED_TRACKS &&
      t->time == 0)
    return;
  time = t->time + TGPS_BASE_TIME;
  tmptr = gmtime(&time);
  strftime(time_buf, sizeof(time_buf), "%F %T GMT", tmptr);

  tgps_print(conn, "T1\t%s\t", time_buf);
  tgps_print_pos(conn, tgps_semicircle_to_deg(t->lat));
  tgps_print(conn, "\t");
  tgps_print_pos(conn, tgps_semicircle_to_deg(t->lon));
  if (t->new_track)
    {
      tgps_print(conn, "%sNewTrack=True", comma);
      comma = ", ";
    }
  if (t->alt > TGPS_INVALID_FLOAT_TEST)
    {
      tgps_print(conn, "%sAlt=%g", comma, t->alt);
      comma = ", ";
    }
  if (t->dpth > TGPS_INVALID_FLOAT_TEST)
    {
      tgps_print(conn, "%sDepth=%g", comma, t->dpth);
      comma = ", ";
    }
  for(i = sizeof(t->extra) - 1; i >= 0; i--)
    if (t->extra[i])
       break;
   if (i != -1)
     {
       int j;

       tgps_print(conn, "%sExtra=0x", comma);
       for(j = 0; j <= i; j++)
	 tgps_print(conn, "%02x", (unsigned char) t->extra[j]);
       comma = ", ";
     }
  tgps_print(conn, "\n");
}

/* Print track point structure */
void tgps_print_track(Tgps conn, TgpsTrack t)
{
  switch (conn->data_version)
    {
    case TGPS_DATA_VERSION_GPSTRANS:
      tgps_print_track_gpstrans(conn, t);
      break;
    case TGPS_DATA_VERSION_TGPS_V1:
      tgps_print_track_v1(conn, t);
      break;
    }
  fflush(conn->output_stream);
}

/* Print track header structure */
void tgps_print_track_header(Tgps conn, TgpsTrackHeader t)
{
  const char *comma = "";

  tgps_print(conn, "N1\t");
  if (t->ident != NULL)
    tgps_print(conn, "%-20s\t", t->ident);
  else
    tgps_print(conn, "\t");
  
  if (t->display)
    tgps_print(conn, "Display=True");
  else
    tgps_print(conn, "Display=False");
  comma = ", ";

  if (t->color != TGPS_INVALID_CHAR)
    {
      tgps_print(conn, "%sColor=%s", comma,
	      tgps_color_to_string(t->color));
      comma = ", ";
    }
  if (t->line_width != 0)
    {
      tgps_print(conn, "%sWidth=%d", comma, t->line_width);
      comma = ", ";
    }
  if (t->fill)
    {
      tgps_print(conn, "%sFill=%d", comma, t->fill);
      comma = ", ";
    }
  
  tgps_print(conn, "\n");
  fflush(conn->output_stream);
}


/* Print almanac structure in gpstrans format */
void tgps_print_almanac_gpstrans(Tgps conn, TgpsAlmanac a)
{
  int id;

  id = a->svid;
  if (id == TGPS_INVALID_CHAR)
    id = 33 - conn->packet_count;
  if (a->wn == 65535)
    return;
  tgps_print(conn,
	  "******** Week %d almanac for PRN-%d ********\n",
	  a->wn, id);
  tgps_print(conn, "ID:                         %02d\n", id);
  tgps_print(conn, "Health:                     %03d\n",
	     (a->hlth != TGPS_INVALID_CHAR ? a->hlth : 0));
  tgps_print(conn, "Eccentricity:              % 1.10E\n",
	     (double) a->e);
  tgps_print(conn, "Time of Applicability(x):   %.4f\n",
	     (double) a->toa);
  tgps_print(conn, "Orbital Inclination(rad):  % 1.10f\n",
	     (double) a->i);
  tgps_print(conn, "Rate of Right Ascen(r/s):  % 1.10E\n",
	     (double) a->odot);
  tgps_print(conn, "SQRT(A)  (m^1/2):          % 1.10f\n",
	     (double) a->sqrta);
  tgps_print(conn, "Right Ascen at TOA(rad):   % 1.10E\n",
	     (double) a->omg0);
  tgps_print(conn, "Argument of Perigee(rad):  % 1.10f\n",
	     (double) a->w);
  tgps_print(conn, "Mean Anom(rad):            % 1.10E\n",
	     (double) a->m0);
  tgps_print(conn, "Af0(s):                    % 1.10E\n",
	     (double) a->af0);
  tgps_print(conn, "Af1(s/s):                  % 1.10E\n",
	     (double) a->af1);
  tgps_print(conn, "week:                       %d\n", a->wn);
  tgps_print(conn, "\n");
}

/* Print almanac structure */
void tgps_print_almanac_v1(Tgps conn, TgpsAlmanac a)
{
  int id;
  int i;

  id = a->svid;
  if (id == TGPS_INVALID_CHAR)
    id = 33 - conn->packet_count;
  tgps_print(conn, "A1\t%d\n", id);
  if (a->wn != TGPS_INVALID_CHAR)
    tgps_print(conn, "A1\t%d\twn\t%d\n", id, a->wn);
  if (a->toa > TGPS_INVALID_FLOAT_TEST)
    tgps_print(conn, "A1\t%d\ttoa\t%g\n", id, (double) a->toa);
  if (a->af0 > TGPS_INVALID_FLOAT_TEST)
    tgps_print(conn, "A1\t%d\taf0\t%g\n", id, (double) a->af0);
  if (a->af1 > TGPS_INVALID_FLOAT_TEST)
    tgps_print(conn, "A1\t%d\taf1\t%g\n", id, (double) a->af1);
  if (a->e > TGPS_INVALID_FLOAT_TEST)
    tgps_print(conn, "A1\t%d\te\t%g\n", id, (double) a->e);
  if (a->sqrta > TGPS_INVALID_FLOAT_TEST)
    tgps_print(conn, "A1\t%d\tsqrta\t%g\n", id, (double) a->sqrta);
  if (a->m0 > TGPS_INVALID_FLOAT_TEST)
    tgps_print(conn, "A1\t%d\tm0\t%g\n", id, (double) a->m0);
  if (a->w > TGPS_INVALID_FLOAT_TEST)
    tgps_print(conn, "A1\t%d\tw\t%g\n", id, (double) a->w);
  if (a->omg0 > TGPS_INVALID_FLOAT_TEST)
    tgps_print(conn, "A1\t%d\tomg0\t%g\n", id, (double) a->omg0);
  if (a->odot > TGPS_INVALID_FLOAT_TEST)
    tgps_print(conn, "A1\t%d\todot\t%g\n", id, (double) a->odot);
  if (a->i > TGPS_INVALID_FLOAT_TEST)
    tgps_print(conn, "A1\t%d\ti\t%g\n", id, (double) a->i);
  if (a->hlth != TGPS_INVALID_CHAR)
    tgps_print(conn, "A1\t%d\thlth\t%d\n", id, a->hlth);
  for(i = sizeof(a->extra) - 1; i >= 0; i--)
    if (a->extra[i])
      break;
  if (i != -1)
     {
       int j;

       tgps_print(conn, "A1\t%d\tExtra\t0x", id);
       for(j = 0; j <= i; j++)
	 tgps_print(conn, "%02x", (unsigned char) a->extra[j]);
       tgps_print(conn, "\n");
     }
}

/* Print almanac structure */
void tgps_print_almanac(Tgps conn, TgpsAlmanac a)
{
  switch (conn->data_version)
    {
    case TGPS_DATA_VERSION_GPSTRANS:
      tgps_print_almanac_gpstrans(conn, a);
      break;
    case TGPS_DATA_VERSION_TGPS_V1:
      tgps_print_almanac_v1(conn, a);
      break;
    }
  fflush(conn->output_stream);
}

/* Print time structure in gpstrans format */
void tgps_print_time_gpstrans(Tgps conn, TgpsTime t)
{
  time_t gc, mc;
  struct tm *tm;
  int d;

  mc = time(NULL);
  tm = localtime(&mc);
  tm->tm_sec = t->second + tm->tm_gmtoff;
  tm->tm_min = t->minute;
  tm->tm_hour = t->hour;
  tm->tm_mday = t->day;
  tm->tm_mon = t->month - 1;
  tm->tm_year = t->year - 1900;

  gc = mktime(tm);

  tgps_print(conn,
	     "Local time determine from GPS-Receiver is:  %s",
	     ctime(&gc));
  tgps_print(conn,
	     "Local time on machine is:                   %s\n",
	     ctime(&mc));
  d = (gc - mc);
  if (d < 0) d = -d;
  tgps_print(conn, "Time difference is %d seconds.\n", d);
}

/* Print time structure */
void tgps_print_time_v1(Tgps conn, TgpsTime t)
{
  tgps_print(conn, "C1\t%d-%02d-%02d %02d:%02d:%02d GMT\n",
	     (int) t->year, (int) t->month, (int) t->day,
	     (int) t->hour, (int) t->minute, (int) t->second);
}


/* Set system time */
void tgps_set_time(Tgps conn, TgpsTime t)
{
  struct tm tm, *tm_ptr;
  struct timeval tv;

  memset(&tm, 0, sizeof(tm));
  memset(&tv, 0, sizeof(tv));

  gettimeofday(&tv, NULL);
  tm_ptr = gmtime(&tv.tv_sec);

  if (conn->verbose)
    {
      tgps_print(conn, "Time before is %04d-%02d-%02d %02d:%02d:%02d GMT\n",
		 tm_ptr->tm_year + 1900, tm_ptr->tm_mon + 1, tm_ptr->tm_mday,
		 tm_ptr->tm_hour, tm_ptr->tm_min, tm_ptr->tm_sec);
    }
  tm_ptr = localtime(&tv.tv_sec);
  tm = *tm_ptr;

  tm.tm_sec = t->second;
  tm.tm_min = t->minute;
  tm.tm_hour = t->hour;
  tm.tm_mday = t->day;
  tm.tm_mon = t->month - 1;
  tm.tm_year = t->year - 1900;

  /* Do I really need to take gmtoff in to account myself. Why there isn't
     mktime that takes utc in instead of localtime. */
  tv.tv_sec = mktime(&tm) + tm.tm_gmtoff;
  settimeofday(&tv, NULL);

  if (conn->verbose)
    {
      /* It seems to take some time before the change of time propagates, so
	 wait for a second to make sure the time has actually changed. */
      sleep(1);
      gettimeofday(&tv, NULL);
      tm_ptr = gmtime(&tv.tv_sec);
      tgps_print(conn, "Time after is %04d-%02d-%02d %02d:%02d:%02d GMT\n",
		 tm_ptr->tm_year + 1900, tm_ptr->tm_mon + 1, tm_ptr->tm_mday,
		 tm_ptr->tm_hour, tm_ptr->tm_min, tm_ptr->tm_sec);
    }
}

/* Print time structure */
void tgps_print_time(Tgps conn, TgpsTime t)
{
  if (conn->operation == TGPS_SET_TIME)
    {
      tgps_set_time(conn, t);
      fflush(conn->output_stream);
      return;
    }
  switch (conn->data_version)
    {
    case TGPS_DATA_VERSION_GPSTRANS:
      tgps_print_time_gpstrans(conn, t);
      break;
    case TGPS_DATA_VERSION_TGPS_V1:
      tgps_print_time_v1(conn, t);
      break;
    }
  fflush(conn->output_stream);
}

/* Print position structure */
void tgps_print_position_v1(Tgps conn, TgpsPosition p)
{
  tgps_print(conn, "P1\t");
  tgps_print_pos(conn, tgps_radians_to_deg(p->lat));
  tgps_print(conn, "\t");
  tgps_print_pos(conn, tgps_radians_to_deg(p->lon));
  tgps_print(conn, "\n");
}

/* Print position structure */
void tgps_print_position(Tgps conn, TgpsPosition p)
{
  switch (conn->data_version)
    {
    case TGPS_DATA_VERSION_GPSTRANS:
    case TGPS_DATA_VERSION_TGPS_V1:
      tgps_print_position_v1(conn, p);
      break;
    }
  fflush(conn->output_stream);
}

/* Print pvt structure */
void tgps_print_pvt_v1(Tgps conn, TgpsPvt p)
{
  tgps_print(conn, "L1\t");
  tgps_print_pos(conn, tgps_radians_to_deg(p->lat));
  tgps_print(conn, "\t");
  tgps_print_pos(conn, tgps_radians_to_deg(p->lon));
  tgps_print(conn, "\t");
  tgps_print(conn, "%.0f\t%.0f\t%.2f,%.2f,%.2f\t%.0f,%.0f,%.0f\t",
	     (double) p->alt + p->msl_hght, (double) p->msl_hght, 
	     (double) p->east, (double) p->north, (double) p->up,
	     (double) p->epe, (double) p->eph, (double) p->epv);
  tgps_print(conn, "%s\t", tgps_fix_to_string(p->fix));
  tgps_print(conn, "%ldd+%gs-%dls\n",
	     p->wn_days, (double) p->tow, p->leap_scnds);
}

/* Print pvt structure */
void tgps_print_pvt(Tgps conn, TgpsPvt p)
{
  switch (conn->data_version)
    {
    case TGPS_DATA_VERSION_GPSTRANS:
    case TGPS_DATA_VERSION_TGPS_V1:
      tgps_print_pvt_v1(conn, p);
      break;
    }
  fflush(conn->output_stream);
}

/* Print header */
void tgps_print_header_v1(Tgps conn)
{
  tgps_print(conn, "V1\t");
  switch (conn->degree_format)
    {
    case TGPS_DEG_MIN_SEC:
      tgps_print(conn, "DMS"); break;
    case TGPS_DEG_MIN:
      tgps_print(conn, "DM"); break;
    case TGPS_DEG:
      tgps_print(conn, "D"); break;
    case TGPS_DEG_NONE:
      fprintf(stderr, "Internal error: Degree format is none\n");
      exit(1);
      break;
    }
  tgps_print(conn, "\n");
}

/* Print header in gpstrans format */
void tgps_print_header_gpstrans(Tgps conn)
{
  struct tm *tmptr;
  time_t t;

  tgps_print(conn, "Format: ");
  switch (conn->degree_format)
    {
    case TGPS_DEG_MIN_SEC:
      tgps_print(conn, "DMS"); break;
    case TGPS_DEG_MIN:
      tgps_print(conn, "DMM"); break;
    case TGPS_DEG:
      tgps_print(conn, "DDD"); break;
    case TGPS_DEG_NONE:
      fprintf(stderr, "Internal error: Degree format is none\n");
      exit(1);
      break;
    }
  t = time(NULL);
  tmptr = localtime(&t);
  tgps_print(conn, "  UTC Offset: %6.2f hrs  Datum[100]: WGS 84\n",
	     tmptr->tm_gmtoff / 3600.0);
}

/* Print header */
void tgps_print_header(Tgps conn)
{
  switch (conn->data_version)
    {
    case TGPS_DATA_VERSION_GPSTRANS:
      tgps_print_header_gpstrans(conn);
      break;
    case TGPS_DATA_VERSION_TGPS_V1:
      tgps_print_header_v1(conn);
      break;
    }
  fflush(conn->output_stream);
}
